// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
//
// TODO(youngki): Implement this file.

#include "device/bluetooth/bluetooth_adapter_win.h"

#include <BluetoothAPIs.h>
#if defined(_WIN32_WINNT_WIN8) && _MSC_VER < 1700
// The Windows 8 SDK defines FACILITY_VISUALCPP in winerror.h.
#undef FACILITY_VISUALCPP
#endif
#include <delayimp.h>
#include <string>
#include "base/bind.h"
#include "base/logging.h"
#include "base/message_loop.h"
#include "base/stringprintf.h"
#include "base/sys_string_conversions.h"
#include "base/threading/thread_restrictions.h"

#pragma comment(lib, "Bthprops.lib")
#pragma comment(lib, "delayimp.lib")

namespace {

const BLUETOOTH_FIND_RADIO_PARAMS bluetooth_adapter_param =
    { sizeof(BLUETOOTH_FIND_RADIO_PARAMS) };

// A frame-based exception handler filter function for a handler for exceptions
// generated by the Visual C++ delay loader helper function.
int FilterVisualCPPExceptions(DWORD exception_code) {
  return HRESULT_FACILITY(exception_code) == FACILITY_VISUALCPP ?
      EXCEPTION_EXECUTE_HANDLER : EXCEPTION_CONTINUE_SEARCH;
}

// Returns true if the machine has a bluetooth stack available. The first call
// to this function will involve file IO, so it should be done on an
// appropriate thread. This function is not threadsafe.
bool HasBluetoothStack() {
  static enum {
    HBS_UNKNOWN,
    HBS_YES,
    HBS_NO,
  } has_bluetooth_stack = HBS_UNKNOWN;

  if (has_bluetooth_stack == HBS_UNKNOWN) {
    base::ThreadRestrictions::AssertIOAllowed();
    HRESULT hr = E_FAIL;
    __try {
      hr = __HrLoadAllImportsForDll("bthprops.cpl");
    } __except(FilterVisualCPPExceptions(::GetExceptionCode())) {
      hr = E_FAIL;
    }
    has_bluetooth_stack = SUCCEEDED(hr) ? HBS_YES : HBS_NO;
  }

  return has_bluetooth_stack == HBS_YES;
}

}  // namespace

namespace device {

const int BluetoothAdapterWin::kPollIntervalMs = 500;

BluetoothAdapterWin::BluetoothAdapterWin()
    : BluetoothAdapter(),
      powered_(false),
      ALLOW_THIS_IN_INITIALIZER_LIST(weak_ptr_factory_(this)) {
}

BluetoothAdapterWin::~BluetoothAdapterWin() {
}

void BluetoothAdapterWin::AddObserver(BluetoothAdapter::Observer* observer) {
  NOTIMPLEMENTED();
}

void BluetoothAdapterWin::RemoveObserver(BluetoothAdapter::Observer* observer) {
  NOTIMPLEMENTED();
}

bool BluetoothAdapterWin::IsPresent() const {
  return !address_.empty();
}

bool BluetoothAdapterWin::IsPowered() const {
  return powered_;
}

void BluetoothAdapterWin::SetPowered(
    bool powered,
    const base::Closure& callback,
    const ErrorCallback& error_callback) {
  NOTIMPLEMENTED();
}

bool BluetoothAdapterWin::IsDiscovering() const {
  NOTIMPLEMENTED();
  return false;
}

void BluetoothAdapterWin::SetDiscovering(
    bool discovering,
    const base::Closure& callback,
    const ErrorCallback& error_callback) {
  NOTIMPLEMENTED();
}

BluetoothAdapter::ConstDeviceList BluetoothAdapterWin::GetDevices() const {
  NOTIMPLEMENTED();
  return BluetoothAdapter::ConstDeviceList();
}

BluetoothDevice* BluetoothAdapterWin::GetDevice(const std::string& address) {
  NOTIMPLEMENTED();
  return NULL;
}

const BluetoothDevice* BluetoothAdapterWin::GetDevice(
    const std::string& address) const {
  NOTIMPLEMENTED();
  return NULL;
}

void BluetoothAdapterWin::ReadLocalOutOfBandPairingData(
    const BluetoothOutOfBandPairingDataCallback& callback,
    const ErrorCallback& error_callback) {
  NOTIMPLEMENTED();
}

void BluetoothAdapterWin::UpdateAdapterState() {
  // TODO(youngki): Move this check to the earliest point reasonable so that no
  // attempts are made to do any bluetooth processing if no BT stack is present.
  if (!HasBluetoothStack())
    return;

  HBLUETOOTH_RADIO_FIND bluetooth_adapter_handle = NULL;
  BLUETOOTH_RADIO_INFO bluetooth_adapter_info =
      { sizeof(BLUETOOTH_RADIO_INFO), 0 };
  HBLUETOOTH_RADIO_FIND bluetooth_handle = BluetoothFindFirstRadio(
      &bluetooth_adapter_param, &bluetooth_adapter_handle);

  if (bluetooth_adapter_handle) {
    if (ERROR_SUCCESS == BluetoothGetRadioInfo(bluetooth_adapter_handle,
                                               &bluetooth_adapter_info)) {
      name_ = base::SysWideToUTF8(bluetooth_adapter_info.szName);
      address_ = base::StringPrintf("%02X:%02X:%02X:%02X:%02X:%02X",
          bluetooth_adapter_info.address.rgBytes[5],
          bluetooth_adapter_info.address.rgBytes[4],
          bluetooth_adapter_info.address.rgBytes[3],
          bluetooth_adapter_info.address.rgBytes[2],
          bluetooth_adapter_info.address.rgBytes[1],
          bluetooth_adapter_info.address.rgBytes[0]);
      powered_ = BluetoothIsConnectable(bluetooth_adapter_handle) ||
          BluetoothIsDiscoverable(bluetooth_adapter_handle);
    } else {
      name_.clear();
      address_.clear();
      powered_ = false;
    }
  }

  if (bluetooth_handle)
    BluetoothFindRadioClose(bluetooth_handle);
}

void BluetoothAdapterWin::TrackDefaultAdapter() {
  PollAdapterState();
}

void BluetoothAdapterWin::PollAdapterState() {
  UpdateAdapterState();

  MessageLoop::current()->PostDelayedTask(
      FROM_HERE,
      base::Bind(&BluetoothAdapterWin::PollAdapterState,
                 weak_ptr_factory_.GetWeakPtr()),
      base::TimeDelta::FromMilliseconds(kPollIntervalMs));
}

}  // namespace device
