#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

from terra.core.manager import Manager
from list import ListPanel, ListPanelController

mger = Manager()
BaseRowRendererWidget = mger.get_class("Widget/BaseRowRenderer")
RowRendererWidget = mger.get_class("Widget/RowRenderer")


class CheckListRenderer(RowRendererWidget):
    terra_type = "Widget/CheckListRenderer"
    row_group = "panel_list_item_check"

    def __init__(self, parent, theme=None):
        RowRendererWidget.__init__(self, parent, theme)
        self.transition = False

    def _is_selected(self, v):
        return v.checked

    def do_value_set(self, v):
        if self.transition:
            suffix = ",transition"
            self.transition = False
        else:
            suffix = ""

        self.part_text_set("text", v.name)
        if self._is_selected(v):
            self.signal_emit("item,state,checked" + suffix, "")
        else:
            self.signal_emit("item,state,unchecked" + suffix, "")

        self._model = v


class CheckListPanel(ListPanel):
    terra_type = "Widget/CheckListPanel"

    def __init__(self, main_window, title, elements, row_renderer=None,
                 theme=None, header_text=None):
        if row_renderer is None:
            row_renderer = CheckListRenderer
        ListPanel.__init__(self, main_window, title, elements,
                           row_renderer, theme=theme,
                           header_text=header_text)

    def redraw_item(self, index, end_callback=None):
        r = self._list.renderer_for_index(index)
        if r is not None:
            r.transition = True
            r.force_redraw()
        if end_callback:
            end_callback()


class CheckListPanelController(ListPanelController):
    terra_type = "Controller/Settings/CheckedFolder"
    row_renderer = None

    def _setup_view(self):
        title = self.model.name
        self.view = CheckListPanel(self.parent.window, title,
                                   self.model.children, self.row_renderer)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.callback_escape = self.back

    def cb_on_clicked(self, view, index):
        model = self.model.children[index]
        def cb(*ignored):
            self.view.redraw_item(index)
        self.use(model, end_callback=cb)


# TODO: remove completely this Settings vs. Options difference
class OptionsCheckedFolderController(CheckListPanelController):
    terra_type = "Controller/Options/CheckedFolder"


class CheckListSinglePanelController(CheckListPanelController):
    terra_type = "Controller/Settings/CheckedFolderSingle"

    class CheckListSingleRenderer(CheckListRenderer):
        def _is_selected(self, v):
            return v.parent.current is not None and \
                v.parent.children[v.parent.current] is v

    row_renderer = CheckListSingleRenderer

    def __init__(self, model, canvas, parent):
        CheckListPanelController.__init__(self, model, canvas, parent)
        if self.model.current is not None:
            self.view.redraw_item(self.model.current)

    def cb_on_clicked(self, view, index):
        old_index = self.model.current
        self.model.current = index

        if old_index is not None:
            self.view.redraw_item(old_index)

        self.view.redraw_item(index)
        self.use(self.model.children[index], end_callback=None)
