# -*- coding: utf-8 -*-
#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import re
import ecore
import string
import logging
import shutil
import tempfile
import unicodedata

from terra.utils.encoding import to_utf8
from os.path import normpath, basename, isdir

try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite

from terra.core.manager import Manager
from terra.core.model import Model, ModelFolder
from terra.core.plugin_prefs import PluginPrefs
from terra.core.threaded_func import ThreadedFunction
from terra.core.threaded_model import ThreadedModelFolder


log = logging.getLogger("plugins.canola-core.settings.media_library")

mger = Manager()
ModelSettingsFolder = mger.get_class("Model/Settings/Folder")
SystemProperties = mger.get_class("SystemProperties")
SettingsActionModel = mger.get_class("Model/Settings/Action")

def commonpathprefix(paths):
    """Extract the common prefix between the paths in a list.

    Since os.path.commonprefix doesn't treat components of the path
    as the basic unit, we just split the path by os.path.sep and
    call os.path.commonprefix with the splitted paths.

    """
    splitted = [normpath(p).split(os.path.sep) for p in paths]
    common = os.path.commonprefix(splitted)
    return os.path.sep.join(common)


class ModelSettingsMediaLibrary(ThreadedModelFolder):
    """Base model for media folder scanning selection.

    Responsible for reading and write the prefs for with directories will be
    scanned.

    """
    terra_type = "Model/Settings/MediaLibrary"
    title = None
    prefs_key = None
    show_hidden = False

    STATE_NONE, STATE_ALL = range(2)

    def __init__(self, parent):
        ThreadedModelFolder.__init__(self, self.title, parent)
        self.dirs = []
        self.callback_rescan = None

        self.system_props = SystemProperties()
        canola_prefs = PluginPrefs("settings")

        self.paths = self._get_list_from_prefs(canola_prefs, self.prefs_key)
        if not self.paths:
            self.paths = ["/"]

    def _get_list_from_prefs(self, prefs, key):
        try:
            lst = prefs[key]
        except KeyError:
            lst = prefs[key] = []
        return lst

    def do_load(self):
        prefs = PluginPrefs("scanned_folders")
        self.scanned = self._get_list_from_prefs(prefs, self.prefs_key)

        for path in self.paths:
            if os.path.exists(path):
                def is_child(p):
                    return commonpathprefix([path, p]) == normpath(path)
                dir_scanned = filter(is_child, self.scanned)
                t = self.system_props.path_alias_get(path)
                if not t:
                    t = basename(normpath(path))
                ModelSettingsDir(self, path, dir_scanned, t,
                                 show_hidden=self.show_hidden)

    def update_scanned(self):
        new_scanned = []
        for d in self.children:
            new_scanned.extend(d.scanned)

        self.scanned = new_scanned
        self.callback_rescan(self.scanned, self.prefs_key)

    def do_unload(self):
        del self.dirs[:]
        del self.children[:]


class ModelSettingsDir(ThreadedModelFolder):
    terra_type = "Model/Settings/Dir"

    STATE_NONE, STATE_ALL = range(2)

    def __init__(self, parent, path=None, scanned=None, title=None,
                 show_hidden=False):
        self.path = path
        if title is None:
            self.title = basename(self.path)
        else:
            self.title = title

        self.scanned = []
        self._scanned_set(scanned)

        self.dirs = []
        self.loaded_dirs = False
        self._has_subdirs = None
        self.show_hidden = show_hidden

        ThreadedModelFolder.__init__(self, self.title, parent)

    def _scanned_set(self, new_scanned):
        if self.path in new_scanned:
            self.scan_state = self.STATE_ALL
        else:
            self.scan_state_set(self.STATE_NONE, new_scanned)

    def scan_state_get(self):
        return self._scan_state

    def scan_state_set(self, state, new_scanned=None):
        if new_scanned is None:
            new_scanned = []

        del self.scanned[:]
        self._scan_state = state

        if state == self.STATE_ALL:
            self.scanned.append(self.path)
        elif new_scanned is not None:
            self.scanned.extend(new_scanned)

    scan_state = property(scan_state_get, scan_state_set)

    def _get_dirs(self):
        if self.show_hidden:
            def dir_filter(p):
                return isdir(os.path.join(self.path, p))
        else:
            def dir_filter(p):
                return isdir(os.path.join(self.path, p)) and \
                    not p.startswith(".")

        if not self.dirs and not self.loaded_dirs:
            self.loaded_dirs = True
            try:
                self.dirs = [p for p in os.listdir(self.path) if dir_filter(p)]
            except OSError, e:
                log.warning("ModelSettingsDir._get_dirs() %s", e)

        return self.dirs

    def do_load(self):
        dirs = self._get_dirs()
        dirs.sort()

        for dir in dirs:
            full_path = os.path.join(self.path, dir)
            def is_child(p):
                return commonpathprefix([full_path, p]) == full_path
            dir_scanned = filter(is_child, self.scanned)
            ModelSettingsDir(self, full_path, dir_scanned,
                             show_hidden=self.show_hidden)

    def has_subdirs(self):
        """Returns whether this directory has subdirectories."""
        if self._has_subdirs is not None:
            return self._has_subdirs

        self._has_subdirs = False
        files = []

        try:
            files = os.listdir(self.path)
        except OSError, e:
            log.warning("ModelSettingsDir.has_subdirs() %s", e)

        for f in files:
            try:
                if isdir(os.path.join(self.path, f)):
                    self._has_subdirs = True
                    break
            except OSError, e:
                log.warning("ModelSettingsDir.has_subdirs() %s", e)

        return self._has_subdirs

    def update_scanned(self):
        new_scanned = []
        for d in self.children:
            new_scanned.extend(d.scanned)
        self._scanned_set(new_scanned)


class MediaLibraryModel(ModelSettingsFolder):
    terra_type = "Model/Settings/Folder/Root/MediaLibrary"
    title = "Media library"
    children_order = ["/Folders", "/CheckCollections"]
    children_prefixes = ["Model/Settings/Folder/MediaLibrary"]


class MediaFoldersModel(ModelSettingsFolder):
    terra_type = "Model/Settings/Folder/MediaLibrary/Folders"
    title = "Media folders"
    children_order = ["/Audio", "/Photo", "/Video", "/ForceRescan", "/ResetDB"]
    children_prefixes = ["Model/Settings/MediaFolder"]


class MediaLibraryAudioModel(ModelSettingsMediaLibrary):
    terra_type = "Model/Settings/MediaFolder/Audio"
    title = "Audio folders"
    prefs_key = "audio"


class MediaLibraryVideoModel(ModelSettingsMediaLibrary):
    terra_type = "Model/Settings/MediaFolder/Video"
    title = "Video folders"
    prefs_key = "video"


class MediaLibraryPhotoModel(ModelSettingsMediaLibrary):
    terra_type = "Model/Settings/MediaFolder/Photo"
    title = "Photo folders"
    prefs_key = "photo"


class MediaLibraryForceRescan(Model):
    terra_type = "Model/Settings/MediaFolder/ForceRescan"
    name = "Refresh all"

    def __init__(self, parent=None):
        Model.__init__(self, self.name, parent)
        self.callback_locked = None
        self.callback_unlocked = None
        self.callback_killall = None

    def execute(self):
        log.info("Forcing canolad to rescan media.")

        if self.callback_killall:
            self.callback_killall()

        mger.canola_daemon.callbacks_locked.insert(0, lambda: False)
        mger.canola_daemon.callbacks_unlocked.insert(0, self._unlocked_cb)
        mger.canola_daemon.rescan()

        if self.callback_locked:
            self.callback_locked()

    def _unlocked_cb(self):
        log.info("Rescan finished.")
        if self.callback_unlocked:
            self.callback_unlocked()

class MediaLibraryResetDBModel(Model):
    terra_type = "Model/Settings/MediaFolder/ResetDB"
    name = "Reset database"
    hooks_type = "Hook/ResetDB"

    def __init__(self, parent=None):
        Model.__init__(self, self.name, parent)
        self.callback_ready = None

    def execute(self):
        log.info("Forcing database resetting.")

        if self.callback_killall:
            self.callback_killall()

        try:
            classes = mger.get_classes(self.hooks_type)
        except Exception, e:
            log.error("no hooks with type %s found", self.hooks_type)
            return

        mger.canola_db.reconnect()
        con = mger.canola_db.connection

        for cls in classes:
            try:
                obj = cls()
                obj.reset_db(con)
            except Exception, e:
                log.error("plugin %s failed on resetting DB: %s", cls, e)

        log.info("DB resetting finished.")
        if self.callback_ready:
            self.callback_ready()

class StartupSettingsModel(SettingsActionModel):
    terra_type = "Model/Settings/Action/General/StartupScan"
    name = "Refresh media at startup"

    def __init__(self, parent=None):
        SettingsActionModel.__init__(self, parent)
        s = PluginPrefs("settings")
        value = s.get("scan_startup")
        if value is None:
            value = True
        self.checked = value

    def execute(self):
        s = PluginPrefs("settings")
        value = not bool(s.get("scan_startup"))
        s["scan_startup"] = value
        s.save()
        self.checked = value


class AlbumCoverItemModel(object):
    (STATE_UNCHECKED, STATE_CHECKED) = range(2)

    class AlbumArtist(object):
        def __init__(self, id=None, name=None):
            self.id = id
            self.name = name

    class AlbumAudio(object):
        def __init__(self, id, title, trackno):
            self.id = id
            self.title = title
            self.trackno = trackno

    def __init__(self):
        self.id = None
        self.name = None
        self.cover = None
        self.updated = False
        self.normalized_name = None
        self.state = self.STATE_UNCHECKED
        self.audios = []
        self.artist = self.AlbumArtist()

    def audio_add(self, id, title, trackno):
        audio = self.AlbumAudio(id, title, trackno)
        self.audios.append(audio)


class CheckCollectionsFinalModel(ModelFolder):
    terra_type = "Model/Settings/Folder/MediaLibrary/CheckCollectionsFinal"

    final_artist = "Compilation"

    update_audio = """UPDATE audios
                      SET album_id = ? WHERE album_id = ?""";

    update_master = """UPDATE audio_albums
                       SET artist_id = ? WHERE id = ?""";

    insert_artist = """INSERT INTO audio_artists(name) VALUES(?)"""

    select_artist = """SELECT id FROM audio_artists WHERE name = ?"""

    delete_album = """DELETE FROM audio_albums
                      WHERE audio_albums.id = ?"""

    def __init__(self, name, parent=None, albums=None):
        ModelFolder.__init__(self, name, parent)

        self.title = name
        self.albums = albums
        self.folder_name = name
        self.conf = PluginPrefs("settings")

    def do_load(self):
        if len(self.albums) <= 1:
            return

        for c in self.albums:
            self.children.append(c)

    def select_count(self):
        """Return the number of selected items."""
        result = 0
        for c in self.albums:
            if c.state == AlbumCoverItemModel.STATE_CHECKED:
                result += 1
        return result

    def update(self, end_callback=None):
        """Update the database merging the selected albums."""
        try:
            cover_path = self.conf["cover_path"]
        except KeyError:
            cover_path = os.path.join(os.path.expanduser("~"),
                                      ".canola", "covers")

        update_result = self._update_database(mger.canola_db.get_cursor())
        mger.canola_db.commit()

        def update_covers():
            if not update_result:
                return
            master, selected = update_result
            self._update_covers(cover_path, master, selected)

        def update_covers_finished(exception, retval):
            def cb():
                if end_callback:
                    end_callback()
            ecore.timer_add(0.5, cb) # XXX: delay

        ThreadedFunction(update_covers_finished, update_covers).start()

    def _update_database(self, cursor):
        if not self.children:
            return

        master = None
        selected = []
        audio_extend = []
        for c in self.children[:]:
            if c.state == AlbumCoverItemModel.STATE_CHECKED:
                selected.append((c.name, c.artist.name))

                if not master:
                    master = c
                else:
                    cursor.execute(self.update_audio, (master.id, c.id))
                    cursor.execute(self.delete_album, (c.id,))
                    audio_extend.extend(c.audios)

                    # remove from model
                    self.albums.remove(c)
                    self.children.remove(c)

        # nothing to merge
        if not audio_extend:
            return None

        cursor.execute(self.select_artist, (self.final_artist,))
        row = cursor.fetchone()
        if not row:
            # insert compilation
            cursor.execute(self.insert_artist, (self.final_artist,))
            # get compilation id
            cursor.execute(self.select_artist, (self.final_artist,))
            row = cursor.fetchone()

        # update master
        master.artist.id = row[0]
        master.artist.name = self.final_artist
        master.audios.extend(audio_extend)
        cursor.execute(self.update_master, (row[0], master.id))

        return master, selected

    def _update_covers(self, cover_path, master, selected):
        re_cover  = re.compile(r'''cover-[0-9]+.jpg''')
        base_path = os.path.join(cover_path, self.final_artist)
        final_path = os.path.join(base_path, master.name)

        cover_paths = []
        for album_name, artist_name in selected:
            path = os.path.join(cover_path, artist_name, album_name)
            if os.path.exists(path):
                cover_paths.append(path)

        if not os.path.exists(base_path):
            os.makedirs(base_path)

        if not os.path.exists(final_path):
            os.makedirs(final_path)

        mv_path = final_path
        final_path = tempfile.mkdtemp()

        count_thumb = 0
        for cpath in cover_paths:
            lst = os.listdir(cpath)
            lst.sort()
            for file in lst:
                if not re_cover.match(file):
                    continue

                count_thumb += 1
                dst_large = os.path.join(final_path,
                                         "cover-%d.jpg" % count_thumb)
                dst_small = os.path.join(final_path,
                                         "cover-small-%d.jpg" % count_thumb)

                file_num = file[:-4].split('-')[-1]
                src_large = os.path.join(cpath, file)
                src_small = os.path.join(cpath, "cover-small-%s.jpg" % file_num)

                if os.path.exists(src_large):
                    shutil.copy(src_large, dst_large)

                if os.path.exists(src_small):
                    shutil.copy(src_small, dst_small)

        shutil.rmtree(mv_path)
        shutil.copytree(final_path, mv_path)

        for c in ("cover", "cover-small"):
            link = os.path.join(mv_path, c + ".jpg")
            link_src = os.path.join(mv_path, c + "-1.jpg")
            if not os.path.exists(link) and os.path.exists(link_src):
                try:
                    os.symlink(c + "-1.jpg", link)
                except Exception:
                    # copy cover if fail to create link
                    shutil.copy(link_src, link)

                master.cover = link


class CheckCollectionsModel(ModelFolder):
    terra_type = "Model/Settings/Folder/MediaLibrary/CheckCollections"
    title = "Multiple album entries"

    select_album = """SELECT audio_albums.id, audio_albums.name,
                             audio_artists.id, audio_artists.name
                      FROM audio_albums, audio_artists
                      WHERE audio_artists.id = audio_albums.artist_id
                        AND exists(SELECT 1 FROM audios
                                   WHERE audios.album_id = audio_albums.id)"""

    select_audio = """SELECT audios.id, audios.title, audios.trackno
                      FROM audios
                      WHERE audios.album_id = ?"""

    special_table = string.maketrans('!@#$%*()_+=+-[]{}:?<>,.|/\\;~"',
                                     '                             ')

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.callback_on_finished = None

    def normalize_album(self, value):
        """Normalize a string removing accents, special chars
        and duplicate spaces."""
        value = unicodedata.normalize("NFKD", value)
        value = value.encode("ascii", "ignore")
        value = value.translate(self.special_table)
        value = " ".join(value.split()).lower()
        return value

    def check_match(self, a, b):
        """Verify if there's an album match."""
        # full match
        if a.normalized_name == b.normalized_name:
            a.state = AlbumCoverItemModel.STATE_CHECKED
            b.state = AlbumCoverItemModel.STATE_CHECKED
            return True

        return False

    def do_load(self):
        """Load possible duplicate album entries."""
        def load():
            try:
                self._load_final_children(self)
            except sqlite.OperationalError, ex:
                log.error("sqlite error: " + ex.message)

            import time
            time.sleep(2.0) #XXX: delay to view progress

        def load_finished(exception, retval):
            if not self.quit and self.callback_on_finished:
                self.callback_on_finished()

        self.quit = False
        self.func = ThreadedFunction(load_finished, load)
        self.func.start()

    def stop_loading(self):
        """Cancel loading process."""
        self.quit = True

    def _load_final_children(self, main):
        filename = os.path.join(os.path.expanduser("~"),
                                ".canola", "canola.db")

        conn = sqlite.connect(filename)
        conn.text_factory = str
        try:
            cursor = conn.cursor()

            cursor.execute(self.select_album)

            list = []
            for row in cursor.fetchall():
                item = AlbumCoverItemModel()
                item.id = row[0]
                item.name = to_utf8(row[1])
                item.artist.id = row[2]
                item.artist.name = to_utf8(row[3])
                item.duplicate = False
                item.cover = os.path.join(os.path.expanduser("~"), ".canola",
                                          "covers", item.artist.name,
                                          item.name, "cover.jpg")
                item.normalized_name = self.normalize_album(row[1])
                list.append(item)

            length = len(list)
            for i in range(0, length):
                master = list[i]

                if main.quit:
                    conn.close()
                    return

                if master.duplicate:
                    continue

                has_match = False
                duplicate_albums = [master]
                for j in range(i + 1, length):
                    slave = list[j]

                    if main.quit:
                        conn.close()
                        return

                    if self.check_match(master, slave):
                        has_match = True
                        slave.duplicate = True
                        duplicate_albums.append(slave)

                if has_match:
                    for album in duplicate_albums:
                        cursor.execute(self.select_audio, (album.id,))
                        for row in cursor.fetchall():
                            album.audio_add(row[0], to_utf8(row[1]), row[2])

                    item = CheckCollectionsFinalModel(master.name, main,
                                                      duplicate_albums)
        finally:
            conn.close()

