#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import logging
import os

from terra.ui.screen import Screen
from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs
from terra.core.terra_object import TerraObject

from dbus import DBusException

mger = Manager()

ImageFullscreenController = mger.get_class("Controller/Media/Image")
CanolaError = mger.get_class("Model/Notify/Error")
SystemProps = mger.get_class("SystemProperties")

network = mger.get_status_notifier("Network")
system_props = SystemProps()
DownloadManager = mger.get_class("DownloadManager")


download_mger = DownloadManager()

SIZE_REQUIREMENT = 3 * (1024 ** 2) # 3 Mb

log = logging.getLogger("canola.plugins.photocast.ui")


class PhotocastControllerMixin(TerraObject):
    terra_type = "PhotocastControllerMixin"

    def __init__(self):
        if self.view is None or not isinstance(self.view, Screen):
            raise TypeError("You must have a Screen typed view.")

        if self.parent is None:
            raise TypeError("You must have parent.")

        self.is_downloading = False
        if self.model.current is None:
            self.model.current = 0

        self.download_process = None

        self.cache_size = 3
        self.cache_items = []

    def _set_download_target(self, model, filename):
        settings = PluginPrefs("settings")
        path = settings.get('download_path', None)

        if path is None:
            path = system_props.DEFAULT_DIR_IMAGES

        path = os.path.join(path, "Photocasts")

        try:
            system_props.prepare_write_path(path,
                "Download folder", SIZE_REQUIREMENT)
        except Exception, e:
            error = CanolaError(e.message)
            self.parent.propagate_error(self, error)
            return False

        model.path = os.path.join(path, filename)
        return True

    def request_image(self, model):
        self.model_item = model
        model.callback_loaded = self.image_loaded

        if model.path and os.path.exists(model.path):
            self._download_process_finished(None, None)
            return

        filename = os.path.basename(model.uri)
        success = self._set_download_target(model, filename)

        if not success:
            return

        self._download_process_add(model)
        self.is_downloading = True
        self.download_process.start()

        self._cache_add(model.path) # XXX move elsewhere?

    def _download_process_add(self, model):
        self.download_process = \
            download_mger.add(model.uri, model.path)
        self.download_process.on_finished_add(self._download_process_finished)
        self.download_process.on_cancelled_add(self._download_process_cancelled)

    def _download_process_remove(self):
        if self.download_process is None:
            return

        self.download_process.on_finished_remove(self._download_process_finished)
        self.download_process.on_cancelled_remove(self._download_process_cancelled)
        self.download_process = None

    def _cache_add(self, item):
        if len(self.cache_items) == self.cache_size:
            del_item = self.cache_items.pop(0)
            if os.path.exists(del_item):
                try:
                    os.unlink(del_item)
                except OSError, oe:
                    log.warning("While trying to delete %s, %s", del_item, oe)

        self.cache_items.append(item)

    def _cache_clear(self):
        item = ""
        for item in self.cache_items:
            if os.path.exists(item):
                try:
                    os.unlink(item)
                except OSError, oe:
                    log.warning("While trying to delete %s, %s", item, oe)
        try:
             os.rmdir(os.path.dirname(item))
        except OSError, oe:
            pass

    def _handle_error(self, msg):
        self.model.state_reason = CanolaError(msg)
        self.is_downloading = False

    def _calc_image_size(self, path):
        image = self.view.evas.FilledImage()
        image.file_set(path)
        w, h =  image.image_size_get()
        image.delete()
        return w, h

    def _download_process_cancelled(self):
        self._download_process_remove()
        self.is_downloading = False

    def _download_process_finished(self, exception, mimetype):
        self._download_process_remove()
        self.is_downloading = False

        model = self.model_item
        if model is None:
            return

        if exception is not None:
            if self.model.is_valid:
                # TODO: use a more user friendly msg
                self._handle_error(exception.message)
            else:
                log.debug("Not handling exception on invalid model\n" + \
                          exception.message)
            return

        if model.width is None or model.height is None:
            model.width, model.height = self._calc_image_size(model.path)

        if model.callback_loaded is not None:
            model.callback_loaded(model)

    def image_loaded(self, model):
        pass

    def prev(self):
        try:
            prev_model = self.model.prev()
            self.request_image(prev_model)
        except DBusException, de:
            log.debug("While downloading prev: %s", de)
            self.is_downloading = False
            raise IndexError("Problem fetching data for model")

    def next(self, loop):
        try:
            next_model = self.model.next(loop)
            self.request_image(next_model)
        except DBusException, de:
            log.debug("While downloading prev: %s", de)
            self.is_downloading = False
            raise IndexError("Problem fetching data for model")

    def back(self):
        self.model.callback_start = None
        ImageFullscreenController.back(self)

    def delete(self):
        if self.download_process is not None:
            self.download_process.cancel()

        self._cache_clear()


class PhotocastFullscreenController(ImageFullscreenController,
                                    PhotocastControllerMixin):
    terra_type = "Controller/Media/Image/Shared"

    def __init__(self, model, canvas, parent):
        ImageFullscreenController.__init__(self, model, canvas, parent)
        PhotocastControllerMixin.__init__(self)

        self.model.callback_state_changed = self._model_state_changed
        self.resume_slideshow = False

        if not self.is_model_folder:
            self.load_initial()

    def _model_loaded(self, model):
        ImageFullscreenController._model_loaded(self, model)
        if model.is_valid:
            self.load_initial()

    def _show_image(self, *args):
        self.view.throbber_stop()
        self.view.show_image(self.is_slideshow_active())

    def image_loaded(self, model):
        PhotocastControllerMixin.image_loaded(self, model)
        self.view.image_new_set(model, self._show_image)

        if self.resume_slideshow:
            self.resume_slideshow = False
            log.debug("resuming slideshow")
            self.start_slideshow_timeout()

    def _model_state_changed(self, model):
        if not model.is_valid:
            self.parent.propagate_error(self, model.state_reason)

    def load_initial(self):
        self.view.throbber_start()
        self.zoom_update_view_controls()
        self._check_prev_next_visibility()

        img_model = self.model.children[self.model.current]
        self.request_image(img_model)

    def load_prev(self, end_callback=None):
        try:
            self.view.throbber_start()
            PhotocastControllerMixin.prev(self)
        except IndexError, ie:
            log.debug("When clicking prev: %s", ie)
            self._end_reached()
            if not self.is_slideshow_active():
                self.view.throbber_stop()
            return False

        self._check_prev_next_visibility()
        return True

    def load_next(self, loop=False, end_callback=None):
        if self.is_downloading:
            # for slideshow
            log.debug("holding slideshow while download")
            self.resume_slideshow = True
            self.view.throbber_start()
            self._slideshow_timer = None
            return False

        try:
            if not self.is_slideshow_active():
                self.view.throbber_start()
            PhotocastControllerMixin.next(self, loop)
        except IndexError, ie:
            log.debug("When clicking next: %s", ie)
            self._end_reached()
            if not self.is_slideshow_active():
                self.view.throbber_stop()
            return False

        if not loop:
            self._check_prev_next_visibility()
        return True

    def cb_on_transition_in_finished(self, *ignored):
        if not self.first_transition:
            return

        self.first_transition = False
        if self.is_model_folder:
            self._cmd_emitted()
            if self.model.is_valid:
                self.view.throbber_start()

    def delete(self):
        ImageFullscreenController.delete(self)
        PhotocastControllerMixin.delete(self)
