#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import dbus
import ecore
import random
import logging

from ui import PlayerScreen
from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs
from terra.core.terra_object import TerraObject

try:
    from e_dbus import DBusEcoreMainLoop
    DBusEcoreMainLoop(set_as_default=True)
except Exception:
    import dbus.ecore


mger = Manager()
Player = mger.get_class("Player")
CanolaError = mger.get_class("Model/Notify/Error")
OptionsControllerMixin = mger.get_class("OptionsControllerMixin")

class PlayerCurrentMediaURI(TerraObject):
    terra_type = "Current/MediaURI"

    uri = None
    model = None
    callbacks = []

    @classmethod
    def set(cls, uri, model):
        if uri == cls.uri and model is cls.model:
            return
        cls.uri = uri
        cls.model = model
        for c in cls.callbacks:
            c(cls)


class PlayerController(Player, OptionsControllerMixin):
    terra_type = "Controller/Media"
    name = "default"

    # Player state type
    (STATE_NONE, STATE_PLAYING,
     STATE_PAUSED, STATE_ERROR, STATE_HOLD) = range(5)

    # Player errors
    (ERROR_UNKNOWN, ERROR_CREATING_PLAYER, ERROR_STOPPING, ERROR_HOLDING,
     ERROR_REGISTERING_DBUS_NAME, ERROR_PLAYING, ERROR_PLAYER_STOPPED,
     ERROR_RESUMING, ERROR_PLAYER_UNAVAILABLE, ERROR_FILE_NOT_FOUND,
     ERROR_FORMAT_NOT_SUPPORTED, ERROR_PLAYER_GENERIC) = range(12)

    error_msgs = {
        ERROR_UNKNOWN: "Unknown error.",
        ERROR_CREATING_PLAYER: "Error while creating player.",
        ERROR_STOPPING: "Player could not stop.",
        ERROR_HOLDING: "Player could not hold.",
        ERROR_REGISTERING_DBUS_NAME: "Error registering dbus name.",
        ERROR_PLAYING: "Error trying to play.",
        ERROR_PLAYER_STOPPED: "Player stopped for unknown reason.",
        ERROR_RESUMING: "Player could not resume.",
        ERROR_PLAYER_UNAVAILABLE: "Player unavailable.",
        ERROR_FILE_NOT_FOUND: "File not found.",
        ERROR_FORMAT_NOT_SUPPORTED: "Format not supported by player.",
        ERROR_PLAYER_GENERIC: "Player error.",
        }

    PL_DBUS_SERVICE_NAME = "br.org.indt.atabake.PlayerSession"
    PL_DBUS_IFACE        = "br.org.indt.atabake.PlayerSession"

    ME_DBUS_SERVICE_NAME = "br.org.indt.atabake.MediaEngine"
    ME_DBUS_OBJ_PATH     = "/br/org/indt/atabake/MediaEngine"
    ME_DBUS_IFACE        = "br.org.indt.atabake.MediaEngine"

    def __init__(self, model, evas, parent):
        Player.__init__(self, model, evas, parent)
        self.log = logging.getLogger("canola.plugins.canola-core.player")

        self.current = self.model.parent.current

        self.repeat_callback = None
        self.shuffle_callback = None

        self.conf = PluginPrefs("player")
        try:
            self._volume = self.conf["volume"]
        except KeyError:
            self._volume = self.conf["volume"] = 80
            self.conf.save()

        self._repeat = None
        self._shuffle = None
        self.shuffle_idx = None
        self.shuffle_list =  None
        self.repeat_kw = "repeat-%s" % self.name
        self.shuffle_kw = "shuffle-%s" % self.name

        try:
            self.repeat = self.conf[self.repeat_kw]
            self.shuffle = self.conf[self.shuffle_kw]
        except KeyError:
            self.repeat = self.conf[self.repeat_kw] = False
            self.shuffle = self.conf[self.shuffle_kw] = False
            self.conf.save()

        self.pos = 0
        self.duration = None
        self.prev_timer = None
        self.next_timer = None
        self.buffering = None
        self.handle_next = True
        self.state = self.STATE_NONE

        self.view = PlayerScreen(self.evas, self.parent.view)
        self.view.callback_prev = self.prev
        self.view.callback_next = self.next
        self.view.callback_seek = self.seek
        self.view.callback_set_volume = self.set_volume
        self.view.callback_get_duration = self.get_duration
        self.view.callback_update_trackbar = self.update_trackbar
        self.view.callback_handle_play_pause = self.handle_play_pause
        self.view.callback_trackbar_mouse_up = self.trackbar_mouse_up
        self.view.callback_trackbar_mouse_down = self.trackbar_mouse_down

        self.prev_timer = None
        self.next_timer = None
        self.trackbar_updater = None
        self.model.callback_state_changed = self.callback_model_state_changed
        self.model.parent.changed_callback_add(self._reshuffle)
        PlayerCurrentMediaURI.set(self.model.uri, self.model)

        OptionsControllerMixin.__init__(self)

        # should be after setting the initial model
        self.inform_media_changed(self.model)

    def commit_changes(self):
        mger.canola_db.commit()
        self.log.info("Commited info to DB from AudioModel")

    def callback_model_state_changed(self, model):
        if not model.is_valid:
            self.stop()
            self.log.debug_warning("Invalidating model")

    def __set_repeat(self, value):
        self._repeat = self.conf[self.repeat_kw] = value
        if self.repeat_callback:
            self.repeat_callback()

    def __get_repeat(self):
        return self._repeat

    repeat = property(__get_repeat, __set_repeat)

    def _reshuffle(self, model):
        # If needed, trigger a reshuffle when the ModelFolder changes.
        if self.shuffle:
            self.shuffle = True

    def __set_shuffle(self, value):
        self._shuffle = self.conf[self.shuffle_kw] = value
        if value:
            self.shuffle_list = range(len(self.model.parent.children))
            if self.shuffle_list:
                random.seed()
                random.shuffle(self.shuffle_list)
                # move the selected song to first place in list
                first_model = self.shuffle_list[0]
                new_position = self.shuffle_list.index(self.current)
                self.shuffle_list[0] = self.current
                self.shuffle_list[new_position] = first_model
                self.model.parent.current = \
                                          self.shuffle_list.index(self.current)
            else:
                self.model.parent.current = None
        else:
            if self.shuffle_idx is not None:
                self.current = self.model.parent.current = self.shuffle_idx
            self.shuffle_list =  None
        if self.shuffle_callback:
            self.shuffle_callback()

    def __get_shuffle(self):
        return self._shuffle

    shuffle = property(__get_shuffle, __set_shuffle)

    def _reply_handler(self, *args):
        pass

    def _error_handler(self, error):
        self.log.debug_error(error)

    def load_atabake(self):
        self.bus = dbus.SessionBus()
        self.pl_iface = None
        self.me_iface = None
        try:
            self._setup_engine()
            self.me_iface.hold_player()
        except Exception:
            raise

    def _setup_engine(self):
        media_engine_obj = self.bus.get_object(self.ME_DBUS_SERVICE_NAME,
                                               self.ME_DBUS_OBJ_PATH,
                                               introspect=False)

        self.me_iface = dbus.Interface(media_engine_obj,
                                       self.ME_DBUS_IFACE)

        self.player_obj_path = self.me_iface.create_session("")

    def setup_interface(self):
        # resolving unique bus name for player session
        bus_obj = self.bus.get_object("org.freedesktop.DBus",
                                      "/org/freedesktop/DBus",
                                      introspect=False)

        self.serv_name = bus_obj.GetNameOwner(self.PL_DBUS_SERVICE_NAME,
                                              dbus_interface="org.freedesktop.DBus")

        player_obj = self.bus.get_object(self.serv_name,
                                         self.player_obj_path,
                                         introspect=False)
        self.pl_iface = dbus.Interface(player_obj, self.PL_DBUS_IFACE)

        # signals
        self.pl_iface.connect_to_signal("state_signal",
                                        self._player_state_changed)
        self.pl_iface.connect_to_signal("eos_signal",
                                        self._player_eos)
        self.pl_iface.connect_to_signal("error_signal",
                                        self._player_error)
        self.pl_iface.connect_to_signal("buffering_signal",
                                        self._player_buffering)
        try:
            self.state = self.pl_iface.get_state()
        except Exception, e:
            self.log.debug_error("Could not get state: %s", e)
        self.view.set_playing(self.state == self.STATE_PLAYING)

    def play(self):
        try:
            self.pl_iface.play(reply_handler=self._reply_handler,
                               error_handler=self._error_handler)
            self.view.set_playing(True)
        except Exception, e:
            self.log.debug_error("Could not play: %s", e)

    def pause(self):
        try:
            self.pl_iface.pause(reply_handler=self._reply_handler,
                                error_handler=self._error_handler)
            self.view.set_playing(False)
        except Exception, e:
            self.log.debug_error("Could not pause: %s", e)

    def handle_play_pause(self):
        if self.state == self.STATE_PLAYING:
            self.pause()
        else:
            self.play()

    def stop(self):
        try:
            self.pl_iface.stop(reply_handler=self._reply_handler,
                               error_handler=self._error_handler)
            self.view.set_playing(False)
        except Exception, e:
            self.log.debug_error("Could not stop: %s", e)

        self.reset_data()

    def set_fullscreen(self, value):
        try:
            self.pl_iface.set_fullscreen(value,
                                         reply_handler=self._reply_handler,
                                         error_handler=self._error_handler)
        except Exception, e:
            self.log.debug_error("Could not stop: %s", e)

    def set_uri(self, uri, auto=True):
        try:
            uri = dbus.Array(uri, signature=dbus.Signature("y"))
            self.pl_iface.set_uri(uri, auto, reply_handler=self._reply_handler,
                                  error_handler=self._error_handler)
        except Exception, e:
            self.log.debug_error("Could not set uri: %s", e)

    def get_volume(self):
        try:
            self._volume = self.pl_iface.get_volume()
        except Exception, e:
            self.log.debug_error("Could not get volume: %s", e)
            return 0

        return self._volume

    def set_volume(self, value):
        if value > 100.0:
            value = 100.0
        elif value < 0.0:
            value = 0.0

        try:
            self.pl_iface.set_volume(int(value),
                                     reply_handler=self._reply_handler,
                                     error_handler=self._error_handler)
        except Exception, e:
            self.log.debug_error("Could not set volume: %s", e)
            return

        self._volume = value
        self.conf["volume"] = int(value)
        self.inform_volume_changed(value)

    volume = property(get_volume, set_volume)

    def set_video_window(self, value):
        try:
            self.pl_iface.set_video_window(value,
                                           reply_handler=self._reply_handler,
                                           error_handler=self._error_handler)
        except Exception, e:
            self.log.debug_error("Could not set video window: %s", e)

    def seek(self, value):
        try:
            self.pl_iface.seek(value * 1000, reply_handler=self._reply_handler,
                               error_handler=self._error_handler)
            self.inform_position_changed(value)
        except Exception, e:
            self.log.debug_error("Could not seek: %s", e)

    def get_position(self):
        try:
            pos = self.pl_iface.get_position() / 1000
        except Exception, e:
            self.log.debug_error("Could not get position: %s", e)
            return 0.0

        return pos

    def get_duration(self):
        if not self.duration:
            try:
                duration = self.pl_iface.get_duration()
                if duration:
                    self.duration = duration / 1000
                else:
                    return -1

                self.inform_duration(self.duration)
            except Exception, e:
                self.log.debug_error("Could not get duration: %s", e)
                return -1

        return self.duration

    def load_preferences(self, preferences):
        try:
            self.pl_iface.load_preferences(preferences)
        except Exception, e:
            self.log.debug_error("Could not load preferences: %s", e)

    def _change_model(self):
        self.set_uri(self.model.uri)
        self.play()
        return False

    def update_model(self, new_model):
        self.model.callback_state_changed = None
        new_model.callback_state_changed = self.callback_model_state_changed
        self.model = new_model
        PlayerCurrentMediaURI.set(self.model.uri, self.model)
        self.current = self.model.parent.current

        self.inform_media_changed(self.model)

    def _prev_shuffle(self):
        parent = self.model.parent
        current_type = type(self.model)

        # We search ourselves for models of the same type
        while True:
            # This will raise IndexError when we are out of bounds
            parent.prev(self.repeat, same_type=False)

            self.shuffle_idx = self.shuffle_list[parent.current]
            shuffle_type = type(parent.children[self.shuffle_idx])

            if current_type == shuffle_type:
                break

        self.update_model(parent.children[self.shuffle_idx])

    def prev(self, seek_disabled=False):
        if self.prev_timer and self.current > 0:
            self.prev_timer.delete()
        if self.next_timer:
            self.next_timer.delete()

        if self.pos < 3 or seek_disabled:
            try:
                if not self.shuffle:
                    self.update_model(self.model.parent.prev(self.repeat))
                else:
                    self._prev_shuffle()
                self.view.throbber_start()
            except IndexError, e:
                # reach start of list
                pass
            else:
                self._update_trackbar(0.0, 0)
                if self.model:
                    self.stop()
                    self.setup_view()
                    self.prev_timer = ecore.timer_add(0.1, self._change_model)
        else:
            self.seek(0)
            self._update_trackbar(0.0, 0)
            # hack to make the video go to 1st frame
            if self.state == self.STATE_PAUSED:
                self.play()
                self.pause()
        return self.model

    def _next_shuffle(self):
        parent = self.model.parent
        current_type = type(self.model)

        # We search ourselves for models of the same type
        while True:
            # This will raise IndexError when we are out of bounds
            parent.next(self.repeat, same_type=False)

            self.shuffle_idx = self.shuffle_list[parent.current]
            shuffle_type = type(parent.children[self.shuffle_idx])

            if current_type == shuffle_type:
                break

        self.update_model(self.model.parent.children[self.shuffle_idx])

    def next(self):
        up_bound = len(self.model.parent.children) - 1
        if self.next_timer and self.current < up_bound:
            self.next_timer.delete()
        if self.prev_timer:
            self.prev_timer.delete()

        try:
            if not self.shuffle:
                self.update_model(self.model.parent.next(self.repeat))
            else:
                self._next_shuffle()
            self._update_trackbar(0.0, 0)
        except IndexError, e:
            # reach end of list
            self.log.info("Last item on ModelFolder: %s", e)
            self._update_trackbar(0.0, 0)
            self.seek(0)
            self.pos = 0
        else:
            if self.model:
                self.view.throbber_start()
                self.stop()
                self.setup_view()
                self.next_timer = ecore.timer_add(0.1, self._change_model)
        return self.model

    def reset_data(self):
        self.duration = None

    def update_back_trackbar(self, value):
        tb = self.view.part_swallow_get("player_view/track_bar")
        if tb is not None:
            tb.part_drag_value_set("knob_back", value, 0.0)

    def _update_trackbar(self, pos, duration):
        def send_duration(duration):
            tlp.message_send(0, duration)
            tll.message_send(0, duration)

        def send_position(pos):
            tlp.message_send(0, pos)
            tll.message_send(0, pos)

        tb = self.view.part_swallow_get("player_view/track_bar")
        if tb:
            tlp = tb.part_swallow_get("time_label_pos")
            tll = tb.part_swallow_get("time_label_left")
            if pos >= 0.0 and duration >= 0:
                tb.part_drag_value_set("knob", pos, 0.0)
                # set duration and position labels
                send_duration(duration)
                send_position(pos)
            elif pos >= 0.0:
                send_position(pos)

    def disable_trackbar(self):
        duration = 0
        pos = float(self.pos)
        if not self.view.tb_disabled:
            try:
                self.view.change_trackbar_state(enable=False)
            except Exception, e:
                self.log.error("Error disabling trackbar: %s", e)
        return pos

    def update_trackbar(self):
        try:
            self.pos = self.get_position()
        except Exception:
            self.pos = 0.0

        try:
            duration = self.get_duration()
            if duration > 0:
                pos = self.pos / float(duration)
                if self.view.tb_disabled:
                    self.view.change_trackbar_state(enable=True)
            else:
                self.log.debug_error("Duration invalid")
                pos = self.disable_trackbar()
        except Exception, e:
            self.log.error("Error updating trackbar: %s", e)
            pos = self.disable_trackbar()

        self._update_trackbar(pos, duration)
        return True

    def trackbar_mouse_up(self):
        if self.state == self.STATE_PLAYING:
            self.trackbar_updater = ecore.timer_add(1, self.update_trackbar)

    def trackbar_mouse_down(self):
        if self.trackbar_updater:
            self.trackbar_updater.delete()

    def _player_state_changed(self, state):
        self.state = state
        if self.state == self.STATE_PLAYING:
            self.view.throbber_stop()
            self.view.set_playing(True)
            self.trackbar_updater = \
                ecore.timer_add(1, self.update_trackbar)
            self.inform_playing()
        else:
            self.view.set_playing(False)
            if self.trackbar_updater:
                self.trackbar_updater.delete()
            self.inform_paused()
        return True

    def _player_eos(self):
        self.log.debug("Received EOS")
        if self.state != self.STATE_ERROR and self.handle_next:
            self.next()
        return True

    def _player_error(self, error_code):
        if error_code != self.ERROR_PLAYER_UNAVAILABLE:
            self._player_state_changed(self.STATE_ERROR)
            notify = CanolaError(self.error_msgs[error_code])
            self.parent.show_notify(notify)
            self.log.error("Player's Controller Error: %d - %s", error_code,
                           self.error_msgs[error_code])
        return True

    def _player_buffering(self, value):
        self.buffering = value
        self.log.debug("Player's  Buffering: %d", value)
        return True

    def back(self):
        self.handle_next = False
        self.conf.save()
        self.parent.back()

    def go_home(self):
        self.handle_next = False
        self.conf.save()
        self.parent.go_home()

    def must_hold(self):
        return False

    def _disconnect_signal(self, *signal_name):
        try:
            for signal in signal_name:
                self.bus.remove_signal_receiver(None, signal,
                                                self.PL_DBUS_IFACE,
                                                self.serv_name,
                                                self.player_obj_path)
        except dbus.DBusException, e:
            # dbus-python version 0.71 has a bug,
            # always raising exception in signal removing
            pass

    def delete(self):
        self.view.delete()
        self.conf.save()
        PlayerCurrentMediaURI.set(None, None)

        if self.prev_timer:
            self.prev_timer.delete()

        if self.next_timer:
            self.next_timer.delete()

        if self.trackbar_updater:
            self.trackbar_updater.delete()

        self._disconnect_signal("state_signal", "eos_signal",
                                "error_signal", "buffering_signal")
        try:
            self.pl_iface.dispose()
        except Exception, e:
            self.log.debug_error(e)

        OptionsControllerMixin.delete(self)
        self.model.callback_state_changed = None
        self.model.parent.changed_callback_del(self._reshuffle)
        del self.pl_iface
        del self.serv_name
        del self.me_iface
        del self.player_obj_path
        del self.bus

        Player.delete(self)

    def block_controls(self):
        self.view.signal_emit("block_controls", "")

    def unblock_controls(self):
        self.view.signal_emit("unblock_controls", "")
