#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

import os
import logging

try:
    from pysqlite2 import dbapi2 as sqlite
except ImportError:
    from sqlite3 import dbapi2 as sqlite

from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs
from terra.core.model import Model, ModelFolder
from terra.core.task import Task


log = logging.getLogger("plugins.canola-core.images.model")

mger = Manager()
db = mger.canola_db
ModelFolderOrdered = mger.get_class("Model/Folder/Ordered")
DefaultIcon = mger.get_class("Icon")


class ImageLocalIcon(DefaultIcon):
    terra_type = "Icon/Folder/Task/Image/Local"
    icon = "icon/main_item/photos_local"


class ImageLocalModel(Model):
    terra_type = "Model/Media/Image"

    __slots__ = ("id", "index", "path",
                 "width", "height",
                 "thumb_width", "thumb_height",
                 "mtime")

    def __init__(self, name, parent):
        Model.__init__(self, name, parent)
        self.id = None
        self.path = None
        self.width = None
        self.height = None
        self.mtime = None


class ImageLocalModelFolder(ModelFolder):
    terra_type = "Model/Folder/Image"

    # Sub classes may need to override query
    query = "SELECT images.id, title, path, width, height, mtime" \
            " FROM files, images WHERE files.id = images.id AND files.dtime = 0"
    select_order = " ORDER BY images.date"
    query_args = ""

    def __init__(self, name, parent):
        ModelFolder.__init__(self, name, parent)
        self.title = name

    def do_load(self):
        self.size = 0

        try:
            rows = db.execute(self.query + self.select_order, self.query_args)
        except sqlite.OperationalError, oe:
            log.debug("While loading photos from bd, %s", oe)
            return

        for index, row in enumerate(rows):
            item = self._create_item_simple(index, row)
            self.size += 1

    def _create_item_simple(self, index, row):
        path = str(row[2])
        name = row[1].encode('utf-8')
        item = ImageLocalModel(name, self)

        item.index = index
        item.id = row[0]
        item.path = path
        item.width = row[3]
        item.height = row[4]
        item.mtime = row[5]

        return item


class ImageLocalAllModelFolder(ImageLocalModelFolder):
    terra_type = "Model/Folder/Image/All"


class ImageLocalAlbumModelFolder(ImageLocalModelFolder):
    terra_type = "Model/Folder/Image/Album"

    clause_get_from_dir = " AND path LIKE ?"

    def __init__(self, name, dir, parent):
        ImageLocalModelFolder.__init__(self, name, parent)

        self.query += self.clause_get_from_dir
        if not dir.endswith("/"):
            dir = dir + "/"
        self.query_args = (dir + "%",)


class ImageLocalBrowse(ModelFolder):
    terra_type = "Model/Folder/Directory"
    child_cls = mger.get_class("Model/Folder/Directory/Media/Image")
    sys_props = mger.get_class("SystemProperties")()

    def do_load(self):
        prefs = PluginPrefs("scanned_folders")
        if not prefs:
            return

        for path in prefs.get("photo", []):
            if os.path.exists(path):
                name = self.sys_props.path_alias_get(path)
                if not name:
                    name = path
                self.child_cls(path, name, self)


def _db_dirname(buf):
    s = str(buf)
    i = s.rindex('/')
    d = s[:i]
    return buffer(d)


class ImageAlbumModelFolder(ModelFolder):
    terra_type = "Model/Folder/Album/Image"

    SystemProperties = mger.get_class("SystemProperties")
    system_props= SystemProperties()

    def do_load(self):
        mger.canola_db.connection.create_function('dirname', 1, _db_dirname)
        base_dirs = self._load_dirs()

        if not base_dirs:
            return

        basepath_dirs = []
        for dir in base_dirs:
            dir_alias = self.system_props.path_alias_get(dir)
            if dir_alias:
                ImageLocalAlbumModelFolder(dir_alias, dir, self)
            else:
                dir_name = os.path.basename(os.path.normpath(dir))
                basepath_dirs.append((dir_name, dir))

        for dir_name, dir in basepath_dirs:
            ImageLocalAlbumModelFolder(dir_name, dir, self)

    def _load_dirs(self):
        try:
            rows = db.execute('SELECT DISTINCT dirname(path) ' \
                                  'FROM files, images USING (id)')
        except sqlite.OperationalError, oe:
            log.debug("While loading photos from bd, %s", oe)
            return []

        dirs = []
        for r in rows:
            dirs.append(str(r[0]))
        return dirs


class ImageLocalList(ModelFolder, Task):
    terra_type = "Model/Folder/Task/Image/Local"
    terra_task_type = "Task/Image/Local"

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, "My photos", parent)

    def do_load(self):
        ImageLocalAllModelFolder("All photos", self)
        # FIXME: disabling browse by folder for now
        #ImageLocalBrowse("Browse by folder", self)
        ImageAlbumModelFolder("Albums", self)


class PhotosResetDBHook(Model):
    terra_type = "Hook/ResetDB"

    def __init__(self, name=None, parent=None):
        Model.__init__(self, name, parent)

    def reset_db(self, con):
        cmd = ("DELETE FROM files WHERE id IN (SELECT files.id"
                " FROM files,images WHERE files.id = images.id)")

        try:
            log.debug("executing %s over %s", cmd, con)
            con.execute(cmd)
            con.commit()
        except Exception, e:
            log.error("%s db error: %r %s", self.__class__, cmd, e)


