# -*- coding: utf-8 -*-
# Canola2 Tuning Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Authors: Adriano Rezende <adriano.rezende@openbossa.org>
#          Renato Chencarek <renato.chencarek@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.


from terra.core.manager import Manager
from terra.core.model import ModelFolder

from backend import CanolaTuningBackend

manager = Manager()
network = manager.get_status_notifier("Network")


class ScanCoversModel(ModelFolder):
    terra_type = "Model/Settings/Folder/MediaLibrary/Tuning/ScanCovers"
    title = "Update cover art"

    class RowSourceModel(object):
        def __init__(self, name, value, parent):
            self.name = name
            self.checked = True
            self.value = value
            self.parent = parent
            parent.children.append(self)

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.canola_tuning = CanolaTuningBackend()

    def do_load(self):
        self.RowSourceModel("Local covers", "local", self)
        self.RowSourceModel("Embedded covers (ID3)", "id3", self)
        # check network
        if network and network.status > 0.0:
            self.RowSourceModel("Last.fm site", "lastfm", self)
            self.RowSourceModel("Amazon site", "amazon", self)

    def get_checked(self):
        return filter(lambda c: c.checked, self.children)

    def select_count(self):
        return len(self.get_checked())

    def update(self, end_callback, cb_progress):
        sources = [c.value for c in self.get_checked()]
        self.canola_tuning.scan_audios(sources, end_callback, cb_progress)

    def cancel(self):
        self.canola_tuning.cancel()


class ScanCoversItemOptionsModel(ScanCoversModel):
    terra_type = "Model/Options/Folder/Media/Audio/AlbumArt/ScanCovers"
    title = "Update cover art"

    def update(self, artist, album, end_callback=None):
        sources = [c.value for c in self.get_checked()]
        self.canola_tuning.scan_audio(sources, artist, album, end_callback)


class CleanUnusedCoversModel(ModelFolder):
    terra_type = "Model/Settings/Folder/MediaLibrary/Tuning/CleanUnusedCovers"
    title = "Remove unused cover art"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.canola_tuning = CanolaTuningBackend()

    def do_load(self):
        pass

    def update(self, end_callback=None):
        self.canola_tuning.clean(end_callback=end_callback)

    def cancel(self):
        self.canola_tuning.cancel()


class CleanUnusedItemOptionsModel(ModelFolder):
    terra_type = "Model/Options/Folder/Media/Audio/AlbumArt/CleanUnused"
    title = "Remove unused cover art"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.canola_tuning = CanolaTuningBackend()

    def do_load(self):
        pass

    def update(self, path, end_callback=None):
        controller = self.parent.screen_controller
        if controller.model.children:
            album_model = controller.model.children[controller.current]
            self.canola_tuning.clean(album_model.artist,
                                     album_model.name, end_callback)

    def cancel(self):
        self.canola_tuning.cancel()


class VideoPlayerThumbnailOptionsModel(ModelFolder):
    terra_type = "Model/Options/Folder/Player/Video/Thumbnail"
    title = "Update thumbnail"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.canola_tuning = CanolaTuningBackend()

    def do_load(self):
        pass

    def update(self, uri, position, end_callback=None):
        self.canola_tuning.scan_video(uri, position, end_callback)

    def cancel(self):
        self.canola_tuning.cancel()


class VideoThumbnailModel(ModelFolder):
    terra_type = "Model/Settings/Folder/MediaLibrary/Tuning/VideoThumbnail"
    title = "Update video thumbnails"
    screenshot_time = 10

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)
        self.canola_tuning = CanolaTuningBackend()

    def do_load(self):
        pass

    def update(self, end_callback=None, cb_progress=None):
        self.canola_tuning.scan_videos(self.screenshot_time, end_callback, cb_progress)

    def cancel(self):
        self.canola_tuning.cancel()


class TuningModelFolder(ModelFolder):
    terra_type = "Model/Settings/Folder/MediaLibrary/Tuning"
    title = "Cover art and thumbnails"

    def __init__(self, parent=None):
        ModelFolder.__init__(self, self.title, parent)

    def do_load(self):
        ScanCoversModel(self)
        VideoThumbnailModel(self)
        CleanUnusedCoversModel(self)
