#include <glib.h>
#include <glib/gi18n.h>
#include <gst/gst.h>
#include <gtk/gtk.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "bognor-local-queue.h"
#include "bognor-player-bindings.h"

enum {
    PROP_0,
};

#define HORNSEY_DBUS_SERVICE "org.moblin.Hornsey"
#define HORNSEY_DBUS_PATH "/org/moblin/Hornsey"
#define BR_PLAYER_DBUS_IFACE "org.moblin.BognorRegis.Player"

struct _BognorLocalQueuePrivate {
    GstElement *playbin;
    GstState audio_state;
    gboolean audio_set;

    DBusGProxy *proxy;
    DBusGProxy *hornsey;
    gboolean hornsey_running;
    gboolean start_requested;

    guint32 tracker_id;
    gint64 duration;

    GtkRecentManager *recent_manager;
    gboolean error_occured;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BOGNOR_TYPE_LOCAL_QUEUE, BognorLocalQueuePrivate))
G_DEFINE_TYPE (BognorLocalQueue, bognor_local_queue, BOGNOR_TYPE_QUEUE);

static void
bognor_local_queue_finalize (GObject *object)
{
    G_OBJECT_CLASS (bognor_local_queue_parent_class)->finalize (object);
}

static void
bognor_local_queue_dispose (GObject *object)
{
    BognorLocalQueue *queue = (BognorLocalQueue *) object;
    BognorLocalQueuePrivate *priv = queue->priv;

    if (priv->proxy) {
        g_object_unref (priv->proxy);
        priv->proxy = NULL;
    }

    if (priv->hornsey) {
        g_object_unref (priv->hornsey);
        priv->hornsey = NULL;
    }

    G_OBJECT_CLASS (bognor_local_queue_parent_class)->dispose (object);
}

static void
bognor_local_queue_set_property (GObject      *object,
                                 guint         prop_id,
                                 const GValue *value,
                                 GParamSpec   *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
bognor_local_queue_get_property (GObject    *object,
                                 guint       prop_id,
                                 GValue     *value,
                                 GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static gboolean
add_item_to_recent (BognorQueue     *queue,
                    BognorQueueItem *item)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GtkRecentData data;
    gboolean ret;

    data.display_name = NULL;
    data.description = NULL;
    data.mime_type = item->mimetype;
    data.app_name = "Hornsey";
    data.app_exec = "hornsey %u";
    data.groups = NULL;
    data.is_private = FALSE;

    ret = gtk_recent_manager_add_full (priv->recent_manager, item->uri, &data);
    if (ret == FALSE) {
        g_warning ("Error registering recent use of %s\n", item->uri);
    }

    return ret;
}

static void
should_show_audio_as_visual (BognorQueue *queue)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GError *error = NULL;

    if (priv->hornsey_running == FALSE) {
        return;
    }

    if (bognor_queue_get_visual_enabled (queue)) {
        BognorQueueItem *item = bognor_queue_get_current_item (queue);

        g_print ("Showing audio as visual\n");
        if (item == NULL) {
            return;
        }

        if (!org_moblin_BognorRegis_Player_show_uri (priv->hornsey, item->uri,
                                                     item->mimetype, &error)) {
            g_warning ("Error telling Hornsey about %s: %s", item->uri,
                       error->message);
            g_error_free (error);
        }
    }
}

static gboolean
set_audio_uri (BognorQueue     *queue,
               BognorQueueItem *item)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    gboolean ret;
    GstState state;

    if (item == NULL) {
        priv->audio_set = FALSE;
        g_object_set (priv->playbin,
                      "uri", "",
                      NULL);
        return;
    }

    if(priv->error_occured) {
        state = GST_STATE_PLAYING;
        priv->error_occured = FALSE;
    } else {
        gst_element_get_state (priv->playbin, &state, NULL,
                               GST_CLOCK_TIME_NONE);
    }

    gst_element_set_state (priv->playbin, GST_STATE_READY);
    /* Audio is played locally */
    g_object_set (priv->playbin,
                  "uri", item->uri,
                  NULL);
    priv->audio_set = TRUE;
    gst_element_set_state (priv->playbin, state);

    should_show_audio_as_visual (queue);

    return TRUE;
}

static gboolean
set_visual_uri (BognorQueue     *queue,
                BognorQueueItem *item)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GError *error = NULL;
    const char *uri, *mimetype;

    if (priv->hornsey_running == FALSE) {
        return TRUE;
    }

    if (item) {
        uri = item->uri;
        mimetype = item->mimetype;
    } else {
        uri = "";
        mimetype = "";
    }

    if (!org_moblin_BognorRegis_Player_show_uri (priv->hornsey, uri,
                                                 mimetype, &error)) {
        g_warning ("Error telling Hornsey about %s: %s", uri,
                   error->message);
        g_error_free (error);
        return FALSE;
    }

    return TRUE;
}

static gboolean
set_uri (BognorQueue     *queue,
         BognorQueueItem *item)
{
    if (item == NULL) {
        set_audio_uri (queue, item);
        set_visual_uri (queue, item);
        return TRUE;
    } else {
        if (item->type == AUDIO_TYPE) {
            return set_audio_uri (queue, item);
        } else {
            return set_visual_uri (queue, item);
        }
    }

    /* Can't reach here */
    return FALSE;
}

static gboolean
set_audio_playing (BognorQueue *queue,
                   gboolean     playing)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;

    if (priv->audio_set == FALSE) {
        return TRUE;
    }

    if (playing) {
        if (priv->audio_state != GST_STATE_PLAYING) {
            gst_element_set_state (priv->playbin, GST_STATE_PLAYING);
            priv->audio_state = GST_STATE_PLAYING;
        }
    } else {
        gst_element_set_state (priv->playbin, GST_STATE_PAUSED);
        priv->audio_state = GST_STATE_PAUSED;
    }

    return TRUE;
}

static gboolean
set_visual_playing (BognorQueue *queue,
                    gboolean     playing)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GError *error = NULL;

    if (priv->hornsey_running == FALSE) {
        return TRUE;
    }

    if (playing) {
        if (!org_moblin_BognorRegis_Player_play (priv->hornsey, &error)) {
            g_warning ("Error telling Hornsey to play: %s", error->message);
            g_error_free (error);
        }
    } else {
        if (!org_moblin_BognorRegis_Player_stop (priv->hornsey, &error)) {
            g_warning ("Error telling Hornsey to stop: %s", error->message);
            g_error_free (error);
        }
    }

    return TRUE;
}

static gboolean
set_audio_position (BognorQueue *queue,
                    double       position)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GstFormat format = GST_FORMAT_TIME;
    gint64 dur;

    if (gst_element_query_duration (priv->playbin, &format, &dur)) {
        gst_element_seek_simple (priv->playbin, format,
                                 GST_SEEK_FLAG_FLUSH | GST_SEEK_FLAG_KEY_UNIT,
                                 (gint64) (dur * position));
    } else {
        g_warning ("Query duration failed");
    }

    return TRUE;
}

static gboolean
set_visual_position (BognorQueue *queue,
                     double       position)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GError *error = NULL;

    if (priv->hornsey_running == FALSE) {
        return TRUE;
    }

    if (!org_moblin_BognorRegis_Player_set_position (priv->hornsey, position,
                                                     &error)) {
        g_warning ("Error seeking video stream: %s\n", error->message);
        g_error_free (error);
        return FALSE;
    }

    return TRUE;
}

static gboolean
get_audio_position (BognorQueue *queue,
                    double      *position)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GstFormat format = GST_FORMAT_TIME;
    gint64 cur, dur;

    *position = 0.0;

    if (gst_element_query_duration (priv->playbin, &format, &dur)) {
        if (gst_element_query_position (priv->playbin, &format, &cur)) {
            *position = ((double) cur) / (double) dur;
        } else {
            g_warning ("Query position failed");
        }
    } else {
        g_print ("Query duration failed");
    }

    return TRUE;
}

static gboolean
get_visual_position (BognorQueue *queue,
                     double      *position)
{
    return TRUE;
}

static void
start_hornsey (BognorLocalQueue *queue)
{
    BognorLocalQueuePrivate *priv = queue->priv;
    guint32 start_ret;
    GError *error = NULL;

    priv->start_requested = TRUE;
    if (!org_freedesktop_DBus_start_service_by_name (priv->proxy,
                                                     HORNSEY_DBUS_SERVICE, 0,
                                                     &start_ret, &error)) {
        g_warning ("Failed to start hornsey: %s", error->message);
        g_error_free (error);

        priv->hornsey_running = FALSE;
        return;
    }
}

static gboolean
force_visual (BognorQueue *queue)
{
    BognorLocalQueue *local = (BognorLocalQueue *) queue;
    BognorLocalQueuePrivate *priv = local->priv;
    GError *error = NULL;

    if (priv->hornsey == NULL) {
        return TRUE;
    }

    if (priv->hornsey_running == FALSE) {
        g_print ("Forcing visual, but hornsey is not running\n");

        if (priv->start_requested == FALSE) {
            start_hornsey ((BognorLocalQueue *) queue);
        }
        return TRUE;
    }

    if (!org_moblin_BognorRegis_Player_force_visual_mode (priv->hornsey,
                                                          &error)) {
        g_warning ("Error forcing visual mode: %s\n", error->message);
        g_error_free (error);
        return FALSE;
    }

    return TRUE;
}

static void
bognor_local_queue_class_init (BognorLocalQueueClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;
    BognorQueueClass *q_class = (BognorQueueClass *) klass;

    o_class->dispose = bognor_local_queue_dispose;
    o_class->finalize = bognor_local_queue_finalize;
    o_class->set_property = bognor_local_queue_set_property;
    o_class->get_property = bognor_local_queue_get_property;

    q_class->set_uri = set_uri;
    q_class->set_audio_playing = set_audio_playing;
    q_class->set_visual_playing = set_visual_playing;
    q_class->set_audio_position = set_audio_position;
    q_class->get_audio_position = get_audio_position;
    q_class->set_visual_position = set_visual_position;
    q_class->get_visual_position = get_visual_position;
    q_class->force_visual = force_visual;
    q_class->add_item_to_recent = add_item_to_recent;

    g_type_class_add_private (klass, sizeof (BognorLocalQueuePrivate));
}

static gboolean
get_position (gpointer userdata)
{
    BognorQueue *queue = (BognorQueue *) userdata;
    BognorLocalQueue *local = (BognorLocalQueue *) userdata;
    BognorLocalQueuePrivate *priv = local->priv;
    double position = 0.0;
    GstFormat format = GST_FORMAT_TIME;
    gint64 cur, dur;

    if (gst_element_query_duration (priv->playbin, &format, &dur)) {
        if (gst_element_query_position (priv->playbin, &format, &cur)) {
            position = ((double) cur) / (double) dur;
        } else {
            g_warning ("Query position failed");
        }
    } else {
        g_print ("Query duration failed");
    }

    /* FIXME: Replace 0 with AUDIO_TYPE */
    bognor_queue_emit_position_changed (queue, position, 0);
    return TRUE;
}

static gboolean
bus_callback (GstBus     *bus,
              GstMessage *message,
              gpointer    data)
{
    BognorQueue *queue = (BognorQueue *) data;
    BognorQueueItem *item;
    BognorLocalQueue *local = (BognorLocalQueue *) data;
    BognorLocalQueuePrivate *priv = local->priv;
    GstState oldstate, newstate;
    GstFormat format;

    switch (GST_MESSAGE_TYPE (message)) {
    case GST_MESSAGE_ERROR: {
        GError *error;
        char *debug;

        gst_message_parse_error (message, &error, &debug);

        g_warning ("Error while playing: %s: %s", error->message, debug);

        g_error_free (error);
        g_free (debug);

        gst_element_set_state(priv->playbin, GST_STATE_NULL);
        priv->error_occured = TRUE;
        break;
    }

    case GST_MESSAGE_EOS:
        /* Once the local GStreamer queue is done we want the next audio */
        gst_element_set_state (priv->playbin, GST_STATE_READY);
        priv->audio_state = GST_STATE_READY;

        bognor_queue_play_next (queue);

        break;

    case GST_MESSAGE_STATE_CHANGED:
        gst_message_parse_state_changed (message, &oldstate, &newstate, NULL);
        if (newstate == GST_STATE_PLAYING) {
            if (priv->tracker_id == 0) {
                priv->tracker_id = g_timeout_add_seconds (1, get_position,
                                                          queue);
            }
        } else {
            if (priv->tracker_id > 0) {
                g_source_remove (priv->tracker_id);
                priv->tracker_id = 0;
            }
        }
        break;

    default:
        break;
    }

    return TRUE;
}

static void
can_show_visual_changed (DBusGProxy  *proxy,
                         gboolean     can_show,
                         BognorQueue *queue)
{
    bognor_queue_set_visual_enabled (queue, can_show);
}

static void
visual_position_changed (DBusGProxy  *proxy,
                         double       position,
                         BognorQueue *queue)
{
    bognor_queue_emit_position_changed (queue, position, VISUAL_TYPE);
}

static void
visual_uri_completed (DBusGProxy  *proxy,
                      const char  *uri,
                      BognorQueue *queue)
{
    bognor_queue_play_next (queue);
}

static void
sync_hornsey (BognorLocalQueue *local)
{
    BognorQueue *queue = (BognorQueue *) local;
    BognorQueueItem *item;

    item = bognor_queue_get_current_item (queue);
    if (item) {
        gboolean is_playing;

        force_visual (queue);

        is_playing = bognor_queue_get_is_playing (queue);
        set_visual_playing (queue, is_playing);
    }
}

static void
name_owner_changed (DBusGProxy *proxy,
                    const char *name,
                    const char *old_owner,
                    const char *new_owner,
                    gpointer    userdata)
{
    BognorLocalQueue *local = (BognorLocalQueue *) userdata;
    BognorLocalQueuePrivate *priv = local->priv;
    BognorQueueItem *item;

    if (g_str_equal (name, HORNSEY_DBUS_SERVICE) == FALSE) {
        /* Only care about Hornsey */
        return;
    }

    if (new_owner && *new_owner != 0) {
        /* Hornsey has just been owned. */
        priv->hornsey_running = TRUE;

        /* If we requested hornsey to start, then we want
           to display something */
        if (priv->start_requested) {
            /* The start request has been completed */
            priv->start_requested = FALSE;

            g_print ("Hornsey has just been owned and is running\n");
            sync_hornsey (local);
        }

        item = bognor_queue_get_current_item ((BognorQueue *) local);
        if (item) {
            set_visual_uri ((BognorQueue *) local, item);
        }

        return;
    }

    /* @new_owner is empty, so hornsey has left the bus for some reason. */
    g_print ("Hornsey has quit\n");

    priv->hornsey_running = FALSE;
    bognor_queue_set_visual_enabled ((BognorQueue *) local, FALSE);
}

static void
listen_to_hornsey (BognorLocalQueue *queue)
{
    BognorLocalQueuePrivate *priv = queue->priv;
    DBusGConnection *connection;
    GError *error = NULL;
    gboolean can_show = FALSE;

    connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (connection == NULL) {
        g_warning ("Error getting bus: %s", error->message);
        g_error_free (error);

        priv->hornsey = NULL;
        return;
    }

    priv->proxy = dbus_g_proxy_new_for_name (connection,
                                             DBUS_SERVICE_DBUS,
                                             DBUS_PATH_DBUS,
                                             DBUS_INTERFACE_DBUS);
    dbus_g_proxy_add_signal (priv->proxy, "NameOwnerChanged",
                             G_TYPE_STRING, G_TYPE_STRING,
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (priv->proxy, "NameOwnerChanged",
                                 G_CALLBACK (name_owner_changed), queue, NULL);

    priv->hornsey = dbus_g_proxy_new_for_name (connection,
                                               HORNSEY_DBUS_SERVICE,
                                               HORNSEY_DBUS_PATH,
                                               BR_PLAYER_DBUS_IFACE);
    dbus_g_proxy_add_signal (priv->hornsey, "CanShowVisualChanged",
                             G_TYPE_BOOLEAN, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (priv->hornsey, "CanShowVisualChanged",
                                 G_CALLBACK (can_show_visual_changed), queue,
                                 NULL);
    dbus_g_proxy_add_signal (priv->hornsey, "PositionChanged",
                             G_TYPE_DOUBLE, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (priv->hornsey, "PositionChanged",
                                 G_CALLBACK (visual_position_changed), queue,
                                 NULL);
    dbus_g_proxy_add_signal (priv->hornsey, "UriCompleted",
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (priv->hornsey, "UriCompleted",
                                 G_CALLBACK (visual_uri_completed), queue,
                                 NULL);

    /* Is hornsey alive? */
    if (!org_freedesktop_DBus_name_has_owner (priv->proxy, HORNSEY_DBUS_SERVICE,
                                              &priv->hornsey_running, &error)) {
        g_warning ("Error checking if Hornsey is running: %s", error->message);
        g_error_free (error);

        bognor_queue_set_visual_enabled ((BognorQueue *) queue, FALSE);
        priv->hornsey_running = FALSE;
        return;
    }

    if (priv->hornsey_running == FALSE) {
        bognor_queue_set_visual_enabled ((BognorQueue *) queue, FALSE);
        return;
    }

    if (!org_moblin_BognorRegis_Player_can_show_visual (priv->hornsey,
                                                        &can_show,
                                                        &error)) {
        g_warning ("Error checking if Hornsey can play visuals: %s",
                   error->message);
        g_error_free (error);

        bognor_queue_set_visual_enabled ((BognorQueue *) queue, FALSE);
        priv->hornsey_running = FALSE;
    } else {
        bognor_queue_set_visual_enabled ((BognorQueue *) queue, can_show);
        priv->hornsey_running = TRUE;
    }
}

static void
bognor_local_queue_init (BognorLocalQueue *self)
{
    BognorLocalQueuePrivate *priv;
    GstElement *fakesink;
    GstBus *bus;

    bognor_queue_set_name ((BognorQueue *) self, _("Playqueue"));

    priv = self->priv = GET_PRIVATE (self);

    listen_to_hornsey (self);

    priv->playbin = gst_element_factory_make ("playbin", "playbin");
    priv->error_occured = FALSE;
    priv->audio_set = FALSE;

    fakesink = gst_element_factory_make ("fakesink", "video_sink");
    g_object_set (priv->playbin,
                  "video-sink", fakesink,
                  NULL);

    bus = gst_pipeline_get_bus (GST_PIPELINE (priv->playbin));
    gst_bus_add_watch (bus, bus_callback, self);
    gst_object_unref (bus);

    priv->recent_manager = gtk_recent_manager_get_default ();
}

BognorLocalQueue *
bognor_local_queue_new (void)
{
    BognorLocalQueue *q;

    q = g_object_new (BOGNOR_TYPE_LOCAL_QUEUE, NULL);

    return q;
}
