#include <string.h>

#include <glib.h>

#include <libgupnp/gupnp-control-point.h>
#include <libgupnp-av/gupnp-av.h>

#include <bickley/bkl-item.h>
#include <bickley/bkl-item-audio.h>
#include <bickley/bkl-item-image.h>
#include <bickley/bkl-item-video.h>

#include "content-directory-service.h"
#include "bkl-finder-upnp.h"

typedef enum _NodeType {
    NODE_TYPE_ROOT,
    NODE_TYPE_CONTAINER,
    NODE_TYPE_ITEM
} NodeType;

typedef struct _NodeData {
    char *id;
    NodeType type;

    guint update_id;
} NodeData;

struct _BklFinderUPnP {
    BklFinder finder;

    GUPnPDeviceInfo *device_info;
    GUPnPServiceProxy *content_directory;
    GUPnPDIDLLiteParser *parser;

    GQueue *pending;

    int amount_left;
    int start;
};

#undef DEBUG

/* Coherence has both 0 and 1000 as the root node, but then uses 1000
   for parentIDs. Which is broken and makes parentID meaningless but they
   refuse to acknowledge it: http://coherence.beebits.net/ticket/179 */
#define ROOT_ID "0"
#define MAX_AMOUNT_PER_REQUEST 25

#define IS_EMPTY_STRING(s) ((s) == NULL || *(s) == '\0')
static void
bkl_finder_upnp_destroy (BklFinder *finder)
{
    BklFinderUPnP *upnp = (BklFinderUPnP *) finder;

    g_object_unref (upnp->content_directory);
    g_object_unref (upnp->parser);

#if 0
    for (l = upnp->pending; l; l = l->next) {
        g_free (l->data);
    }
    g_slist_free (upnp->pending);
#endif

    g_free (upnp);
}

/* FIXME: This only gets the first <res> with an http-get protocolInfo
   attribute. There may by multiple ones with different mimetypes */
static char *
get_http_get_props (GList *didl_props,
                    char **mimetype,
                    char **protocol_info,
                    char **size)
{
    GList *p;

    for (p = didl_props; p; p = p->next) {
        char *pi;

        pi = gupnp_didl_lite_property_get_attribute (p->data, "protocolInfo");
        if (strncmp (pi, "http-get:", 9) == 0) {
            if (mimetype) {
                char *start, *end;

                start = strchr (pi + 9, ':');
                if (start != '\0' && ++start != '\0') {
                    end = strchr (start, ':');
                    *mimetype = g_strndup (start, end - start);
                } else {
                    *mimetype = NULL;
                }
            }

            if (protocol_info) {
                *protocol_info = pi;
            } else {
                g_free (pi);
            }

            if (size) {
                *size = gupnp_didl_lite_property_get_attribute (p->data,
                                                                "size");
            }

            return gupnp_didl_lite_property_get_value (p->data);
        }

        g_free (pi);
    }

    return NULL;
}

static const char *
get_property_keys (xmlNode    *didl_object,
                   const char *property)
{
    GList *props;
    char *ret = NULL;

    props = gupnp_didl_lite_object_get_property (didl_object, property);
    if (props) {
        GList *p;

        for (p = props; p; p = p->next) {
            char *prop = gupnp_didl_lite_property_get_value (p->data);

            if (prop) {
                ret = prop;
                break;
            }
        }

        g_list_free (props);
    }

    return ret;
}

static void
add_artist_properties (xmlNode      *didl_object,
                       BklItemAudio *item)
{
    GList *props;
    GPtrArray *artists;

    props = gupnp_didl_lite_object_get_property (didl_object, "artist");
    if (props) {
        GList *p;

        artists = g_ptr_array_sized_new (g_list_length (props));
        for (p = props; p; p = p->next) {
            char *prop = gupnp_didl_lite_property_get_value (p->data);

            if (prop) {
                g_ptr_array_add (artists, g_strdup (prop));
            }
        }

        bkl_item_audio_set_artists (item, artists);
        g_list_free (props);
    }
}

static void
parse_image_item (xmlNode      *didl_object,
                  BklItemImage *item,
                  const char   *subclass)
{
    if (strncmp (subclass, ".photo", 6) == 0) {
        /* Mark the item as being a photo */
    } else {
        /* Mark the item as being an image */
    }
}

static void
parse_audio_item (xmlNode      *didl_object,
                  BklItemAudio *item,
                  const char   *subclass)
{
    const char *data;

    if (strncmp (subclass, ".musicTrack", 11) == 0) {
        int number;

        data = get_property_keys (didl_object, "album");
        if (!IS_EMPTY_STRING (data)) {
            bkl_item_audio_set_album (item, data);
        }

        data = get_property_keys (didl_object, "originalTrackNumber");
        if (!IS_EMPTY_STRING (data)) {
            number = atoi (data);
            bkl_item_audio_set_track (item, number);
        }

        add_artist_properties (didl_object, item);
    }

    data = get_property_keys (didl_object, "genre");
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_genre (item, data);
    }

    data = get_property_keys (didl_object, "description");
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_comment (item, data);
    }

    data = get_property_keys (didl_object, "rights");
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_audio_set_copyright (item, data);
    }
}

static void
parse_video_item (xmlNode      *didl_object,
                  BklItemVideo *item,
                  const char   *subclass)
{
    const char *data;

    data = get_property_keys (didl_object, "director");
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_video_set_director (item, data);
    }

    /* FIXME: Should get certificate as well? */
}

static void
parse_didl (BklFinderUPnP *finder,
            xmlNode       *didl_object)
{
    BklFinder *f = (BklFinder *) finder;
    BklItem *item = NULL;
    char *uri, *title, *size = NULL, *mimetype = NULL, *protocol_info = NULL;
    gboolean is_item, is_container;
    GList *didl_props;

    title = gupnp_didl_lite_object_get_title (didl_object);
#ifdef DEBUG
    g_print ("(%s): Found %s\n", f->source->name, title);
#endif

    is_item = gupnp_didl_lite_object_is_item (didl_object);
    is_container = gupnp_didl_lite_object_is_container (didl_object);

    if (is_item) {
        char *ref_id;

        ref_id = gupnp_didl_lite_item_get_ref_id (didl_object);
        if (ref_id != NULL) {
#ifdef DEBUG
            g_print ("(%s): %s is just a reference to %s\n",
                     f->source->name, title, ref_id);
#endif
            g_free (title);
            g_free (ref_id);
            return;
        }
    }

    didl_props = gupnp_didl_lite_object_get_property (didl_object, "res");
    uri = get_http_get_props (didl_props, &mimetype, &protocol_info, &size);
    g_list_free (didl_props);

    if (uri == NULL) {
        /* Didn't get any URI we like, so just quit */
        g_free (title);
        return;
    }

#ifdef DEBUG
    g_print ("   at %s\n", uri);
    g_print ("   %s bytes\n", size);
    g_print ("   %s\n", mimetype);
    g_print ("   %s\n", protocol_info);
#endif
    if (is_container) {
        char *class;

        class = gupnp_didl_lite_object_get_upnp_class (didl_object);
        if (strncmp (class, "object.container.", 17) == 0) {
            char *subclass = class + 17;

            if (strncmp (subclass, "album", 5) == 0) {
                /* parse_album_container (didl_object, uri, subclass + 5); */
            }
        }

        g_free (class);
    } else {
        char *class;

        class = gupnp_didl_lite_object_get_upnp_class (didl_object);

        if (strncmp (class, "object.item.", 12) == 0) {
            char *subclass = class + 12;

            if (strncmp (subclass, "imageItem", 9) == 0) {
                BklItemImage *image = bkl_item_image_new ();

                bkl_item_image_set_title (image, title);
                parse_image_item (didl_object, image, subclass + 9);

                item = (BklItem *) image;
            } else if (strncmp (subclass, "audioItem", 9) == 0) {
                BklItemAudio *audio = bkl_item_audio_new ();

                bkl_item_audio_set_title (audio, title);
                parse_audio_item (didl_object, audio, subclass + 9);

                item = (BklItem *) audio;
            } else if (strncmp (subclass, "videoItem", 9) == 0) {
                BklItemVideo *video = bkl_item_video_new ();

                bkl_item_video_set_title (video, title);
                parse_video_item (didl_object, video, subclass + 9);

                item = (BklItem *) video;
            } else if (strncmp (subclass, "playlistItem", 12) == 0) {
                /*parse_playlist_item (didl_object, uri, subclass + 12); */
            } else if (strncmp (subclass, "textItem", 8) == 0) {
                /*parse_text_item (didl_object, uri, subclass + 8);*/
            } else {
                g_warning ("Unknown class: %s", class);
            }
        }

        g_free (class);
    }

    if (item != NULL) {
        GError *error = NULL;

        bkl_item_set_mimetype (item, mimetype);
        bkl_item_set_uri (item, uri);
        if (size) {
            bkl_item_set_size (item, strtol (size, NULL, 10));
        } else {
            bkl_item_set_size (item, 0);
        }

        bkl_source_add_item (f->source, uri, item, &error);
        if (error != NULL) {
            g_warning ("(%s): Error adding %s to DB: %s", f->source->name,
                       uri, error->message);
            g_error_free (error);
        }

        g_object_unref (item);
    }

    g_free (uri);
    g_free (title);
    g_free (size);
    g_free (protocol_info);
}

static void
didl_callback (GUPnPDIDLLiteParser *parser,
               xmlNode             *didl_object,
               gpointer             userdata)
{
    BklFinderUPnP *finder = userdata;
    char *id = gupnp_didl_lite_object_get_id (didl_object);

    /* If the node is a container then we browse it as well */
    if (gupnp_didl_lite_object_is_container (didl_object)) {
        g_queue_push_tail (finder->pending, id);
    }

    /* Parse this didl_object somehow... */
    parse_didl (finder, didl_object);
}

/* Returns TRUE if there is more to browse in this ID,
   FALSE if we're finished */
static gboolean
browse_id (BklFinderUPnP *finder,
           const char    *id)
{
    guint n_ret, n_match, update_id;
    char *didl_result;
    GError *error = NULL;
    gboolean ret;

    int amount_required = finder->amount_left < 0 ? MAX_AMOUNT_PER_REQUEST :
        MIN (finder->amount_left, MAX_AMOUNT_PER_REQUEST);

    ret = cd_browse (finder->content_directory, id, "BrowseDirectChildren",
                     "*", finder->start, amount_required, "", &didl_result,
                     &n_ret, &n_match, &update_id, &error);
    if (ret == FALSE) {
        /* can return FALSE without error -- a gupnp bug? */
        if (!error) {
            g_warning ("Error browsing %s (and no GError set)", id);
        } else {
            g_warning ("Error browsing %s: %s", id, error->message);
            g_error_free (error);
            error = NULL;
        }

        finder->start = 0;
        finder->amount_left = -1;
        return FALSE;
    }

    gupnp_didl_lite_parser_parse_didl (finder->parser, didl_result,
                                       didl_callback, finder, &error);
    if (error != NULL) {
        g_warning ("Error parsing didl: %s", error->message);
        g_error_free (error);

        finder->start = 0;
        finder->amount_left = -1;
        return FALSE;
    }

    if (finder->start == 0 && finder->amount_left == -1) {
        finder->amount_left = n_match - n_ret;
    } else {
        finder->amount_left -= n_ret;
    }

    if (finder->amount_left <= 0) {
        /* Finished with this ID, so reset */
        finder->start = 0;
        finder->amount_left = -1;
        return FALSE;
    } else {
        finder->start += n_ret;
        return TRUE;
    }
}

static gboolean
bkl_finder_upnp_lazy_dig (BklFinder *finder)
{
    BklFinderUPnP *upnp = (BklFinderUPnP *) finder;
    gboolean more = TRUE;
    char *id;

    if (g_queue_is_empty (upnp->pending)) {
        return FALSE;
    }

    id = g_queue_peek_head (upnp->pending);

    more = browse_id (upnp, id);
    if (more == FALSE) {
        /* FIXME: Should we free id here? */
        g_queue_pop_head (upnp->pending);
    }

    return (!g_queue_is_empty (upnp->pending));
}

BklFinder *
bkl_finder_upnp_new (BklSource         *source,
                     GUPnPDeviceProxy  *proxy)
{
    BklFinderUPnP *upnp;
    BklFinder *finder;
    GUPnPDeviceInfo *info = GUPNP_DEVICE_INFO (proxy);

    upnp = g_new0 (BklFinderUPnP, 1);
    finder = (BklFinder *) upnp;

    finder->destroy = bkl_finder_upnp_destroy;
    finder->lazy_dig = bkl_finder_upnp_lazy_dig;

    finder->source = source;

    upnp->parser = gupnp_didl_lite_parser_new ();
    upnp->device_info = info;
    upnp->content_directory =
        (GUPnPServiceProxy *) gupnp_device_info_get_service
        (info, "urn:schemas-upnp-org:service:ContentDirectory:1");

    /* upnp->id_to_node = g_hash_table_new (g_str_hash, g_str_equal); */

    upnp->pending = g_queue_new ();
    g_queue_push_head (upnp->pending, g_strdup (ROOT_ID));

    return finder;
}
