/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>

#include "kozo-pack.h"

#define ALIGN_VALUE(xthis, xalignment) \
        (((xthis) + ((xalignment) - 1)) & (~((xalignment) - 1)))

void
kozo_pack_start (KozoPackContext *c)
{
        c->cur_idx = 0;

        c->length = (c->n_fields + 1) * sizeof (guint32);
        if (c->prepend_count)
                c->length += sizeof (guint32);
}

G_INLINE_FUNC void
finish_pack (KozoPackContext *c)
{
        c->entries[c->cur_idx].offset = ALIGN_VALUE (c->length, c->alignment);

        c->length = c->entries[c->cur_idx].offset +
                    c->entries[c->cur_idx].length;

        c->cur_idx++;
}

void
kozo_pack_data (KozoPackContext *c,
                guint8          *data,
                guint32          length)
{
        g_print ("Packing %p, %d\n", data, length);
        c->entries[c->cur_idx].data = data;
        c->entries[c->cur_idx].length = length;

        finish_pack (c);
}

void
kozo_pack_field (KozoPackContext *c,
                 KozoField       *field)
{
        c->entries[c->cur_idx].data =
                (guint8 *) kozo_field_get_raw_data
                        (field, &(c->entries[c->cur_idx].length));

        finish_pack (c);
}

#define EMPTY_FIELD_LEN 4
const guint8 empty_field[EMPTY_FIELD_LEN] = { 0, 0, 0, 0 };

void
kozo_pack_empty_field (KozoPackContext *c)
{
        c->entries[c->cur_idx].data = (guint8 *) empty_field;
        c->entries[c->cur_idx].length = EMPTY_FIELD_LEN;

        finish_pack (c);
}

void
kozo_pack_string (KozoPackContext *c,
                  const char      *str)
{
        c->entries[c->cur_idx].data = (guint8 *) str;
        c->entries[c->cur_idx].length = strlen (str) + 1;

        finish_pack (c);
}

void
kozo_pack_int (KozoPackContext *c,
               gint32          *i)
{
        c->entries[c->cur_idx].data = (guint8 *) i;
        c->entries[c->cur_idx].length = sizeof (gint32);

        finish_pack (c);
}

void
kozo_pack_time (KozoPackContext *c,
                time_t          *t)
{
        c->entries[c->cur_idx].data = (guint8 *) t;
        c->entries[c->cur_idx].length = sizeof (time_t);

        finish_pack (c);
}

void
kozo_pack_finish (KozoPackContext *c)
{
        guint offset_pos; /* Position of the next offset */

        if (c->prepend_count) {
                /* Prepend field count if requested */
                memcpy (c->dest, (guint8 *) &(c->n_fields), sizeof (guint32));

                offset_pos = sizeof (guint32);
        } else
                offset_pos = 0;

        if (G_LIKELY (c->n_fields > 0)) {
                guint end_pos; /* End position of the previous entry */

                end_pos = c->entries[0].offset;

                for (c->cur_idx = 0; c->cur_idx < c->n_fields; c->cur_idx++) {
                        /* Copy offset */
                        memcpy (c->dest + offset_pos,
                                (guint8 *) &(c->entries[c->cur_idx].offset),
                                sizeof (guint32));

                        /* Pad area between end of previous entry and
                         * beginning of new entry with zeroes. */
                        memset (c->dest + end_pos,
                                0,
                                c->entries[c->cur_idx].offset - end_pos);

                        /* Copy data */
                        memcpy (c->dest + c->entries[c->cur_idx].offset,
                                c->entries[c->cur_idx].data,
                                c->entries[c->cur_idx].length);

                        /* Update counters */
                        end_pos = c->entries[c->cur_idx].offset +
                                  c->entries[c->cur_idx].length;
                        offset_pos += sizeof (guint32);
                }
        }

        /* This indicates where the entry ends */
        memcpy (c->dest + offset_pos,
                (guint8 *) &c->length, sizeof (guint32));
}
