/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "bkl-marshal.h"
#include "bkl-source.h"

enum {
    ADDED,
    DELETED,
    CHANGED,
    IN_PROGRESS,
    COMPLETED,
    INDEX_CHANGED,
    LAST_SIGNAL
};

G_DEFINE_TYPE (BklSource, bkl_source, G_TYPE_OBJECT);

static guint32 signals[LAST_SIGNAL] = {0, };

static gboolean bkl_source_update_uri (BklSource  *source,
                                       const char *IN_uri,
                                       GHashTable *IN_metadata,
                                       GError    **error);
static gboolean bkl_source_get_db_path (BklSource   *source,
                                        const char **OUT_uri,
                                        GError     **error);

#include "bkl-source-glue.h"

static gboolean
bkl_source_update_uri (BklSource  *source,
                       const char *IN_uri,
                       GHashTable *IN_metadata,
                       GError    **error)
{
    return TRUE;
}

static gboolean
bkl_source_get_db_path (BklSource   *source,
                        const char **OUT_uri,
                        GError     **error)
{
    *OUT_uri = g_strdup (kozo_db_get_filename (source->db->db));
    return TRUE;
}

static void
bkl_source_class_init (BklSourceClass *klass)
{
    dbus_g_object_type_install_info (G_TYPE_FROM_CLASS (klass),
                                     &dbus_glib_bkl_source_object_info);
    signals[CHANGED] = g_signal_new ("uri-changed",
                                     G_TYPE_FROM_CLASS (klass),
                                     G_SIGNAL_RUN_FIRST |
                                     G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                     g_cclosure_marshal_VOID__STRING,
                                     G_TYPE_NONE, 1,
                                     G_TYPE_STRING);
    signals[ADDED] = g_signal_new ("uri-added",
                                   G_TYPE_FROM_CLASS (klass),
                                   G_SIGNAL_RUN_FIRST |
                                   G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                   g_cclosure_marshal_VOID__STRING,
                                   G_TYPE_NONE, 1,
                                   G_TYPE_STRING);
    signals[DELETED] = g_signal_new ("uri-deleted",
                                     G_TYPE_FROM_CLASS (klass),
                                     G_SIGNAL_RUN_FIRST |
                                     G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                     g_cclosure_marshal_VOID__STRING,
                                     G_TYPE_NONE, 1,
                                     G_TYPE_STRING);
    signals[IN_PROGRESS] = g_signal_new ("in-progress",
                                         G_TYPE_FROM_CLASS (klass),
                                         G_SIGNAL_RUN_FIRST |
                                         G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                         g_cclosure_marshal_VOID__VOID,
                                         G_TYPE_NONE, 0);
    signals[COMPLETED] = g_signal_new ("complete",
                                       G_TYPE_FROM_CLASS (klass),
                                       G_SIGNAL_RUN_FIRST |
                                       G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                       g_cclosure_marshal_VOID__VOID,
                                       G_TYPE_NONE, 0);
    signals[INDEX_CHANGED] = g_signal_new ("index-changed",
                                           G_TYPE_FROM_CLASS (klass),
                                           G_SIGNAL_RUN_FIRST |
                                           G_SIGNAL_NO_RECURSE, 0, NULL, NULL,
                                           bkl_marshal_VOID__POINTER_POINTER,
                                           G_TYPE_NONE, 2, G_TYPE_STRV,
                                           G_TYPE_STRV);
}

static void
bkl_source_init (BklSource *source)
{
}

gboolean
bkl_source_do_work (BklSource *source)
{
    BklSourceClass *klass = BKL_SOURCE_GET_CLASS (source);

    if (klass->do_work) {
        return klass->do_work (source);
    }

    return FALSE;
}

void
bkl_source_investigate_files (BklSource *source,
                              GPtrArray *files)
{
    BklSourceClass *klass = BKL_SOURCE_GET_CLASS (source);

    if (klass->investigate_files) {
        klass->investigate_files (source, files);
    }
}

void
bkl_source_remove_files (BklSource *source,
                         GPtrArray *files)
{
    BklSourceClass *klass = BKL_SOURCE_GET_CLASS (source);

    if (klass->remove_files) {
        klass->remove_files (source, files);
    }
}

gboolean
bkl_source_add_item (BklSource  *source,
                     const char *uri,
                     BklItem    *item,
                     GError    **error)
{
    BklSourceClass *klass = BKL_SOURCE_GET_CLASS (source);

    if (klass->add_item) {
        return klass->add_item (source, uri, item, error);
    }

    return FALSE;
}

void
bkl_source_uri_changed (BklSource  *source,
                        const char *uri)
{
    g_signal_emit (source, signals[CHANGED], 0, uri);
}

void
bkl_source_uri_added (BklSource  *source,
                      const char *uri)
{
    g_signal_emit (source, signals[ADDED], 0, uri);
}

void
bkl_source_uri_deleted (BklSource  *source,
                        const char *uri)
{
    g_signal_emit (source, signals[DELETED], 0, uri);
}

void
bkl_source_in_progress (BklSource *source)
{
    g_signal_emit (source, signals[IN_PROGRESS], 0);
}

void
bkl_source_completed (BklSource *source)
{
    g_signal_emit (source, signals[COMPLETED], 0);
}

void
bkl_source_index_changed (BklSource *source,
                          GPtrArray *added,
                          GPtrArray *removed)
{
    if (added->len == 0 ||
        added->pdata[added->len - 1] != NULL) {
        g_ptr_array_add (added, NULL);
    }

    if (removed->len == 0 ||
        removed->pdata[removed->len - 1] != NULL) {
        g_ptr_array_add (removed, NULL);
    }

    g_signal_emit (source, signals[INDEX_CHANGED], 0, added->pdata,
                   removed->pdata);
}
