/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <string.h>
#include <errno.h>
#include <signal.h>

#include <glib.h>
#include <kozo.h>
#include <kozo-util.h>

#include <gconf/gconf-client.h>

#include <libgupnp/gupnp-context.h>
#include <libgupnp/gupnp-control-point.h>
#include <libgupnp/gupnp-device-info.h>

#include <dbus/dbus-protocol.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <bickley/bkl-db.h>
#include <bickley/bkl-dbus.h>
#include <bickley/bkl-entry.h>
#include <bickley/bkl-item-audio.h>
#include <bickley/bkl-item-broken.h>
#include <bickley/bkl-item-image.h>
#include <bickley/bkl-item-video.h>
#include <bickley/bkl-utils.h>

#include "bkl-orbiter.h"
#include "bkl-orbiter-dbus.h"
#include "bkl-orbiter-bindings.h"
#include "bkl-path-finder.h"
#include "bkl-investigator-bindings.h"
#include "bkl-marshal.h"
#include "bkl-source-gconf.h"
#include "bkl-source-removable.h"
#include "bkl-source-upnp.h"
#include "bkl-source-manager.h"

#include "metadata-defines.h"

struct _PendingUri {
    BklSource *source;
    char *uri;
};

struct _BklOrbiter {
    BklOrbiterDBus *dbus;
    BklSourceManager *manager;
    BklDBus *bkl_dbus;
    DBusGConnection *connection;
    DBusGProxy *proxy;

    GConfClient *gconf;
    GMainLoop *mainloop;

    GVolumeMonitor *volume_monitor;

    BklSource *local_source;
    GHashTable *sources;
    GHashTable *removable_by_path;

    guint32 worker_id;

    /* These are the uris that we've told the investigator to look at
       and it hasn't got to yet */
    GHashTable *requested_uris;
    char *processing_uri;

    DBusGProxy *investigator;
    GPid investigator_pid;
    guint32 investigator_processing_id;

    GUPnPContext *upnp_monitor;
    GUPnPControlPoint *upnp_cp;
};

#define ORBITER_NAME "org.moblin.Bickley.Orbiter"
#define ORBITER_PATH "/org/moblin/Bickley/Orbiter"
#define ORBITER_IFACE "org.moblin.Bickley.Orbiter"

#define INVESTIGATOR_NAME "org.moblin.Bickley.Investigator"
#define INVESTIGATOR_PATH "/org/moblin/Bickley/Investigator"
#define INVESTIGATOR_IFACE "org.moblin.Bickley.Investigator"

#define BKL_SOURCE_MANAGER_PATH "/org/moblin/Bickley/SourceManager"
#define BKL_LOCAL_SOURCE_PATH "/org/moblin/Bickley/Sources/Local"

#define PROCESSING_ALLOWED_TIMEOUT 10 /* Seconds */
#define INITIAL_SLEEP_TIMEOUT 60 /* Seconds */

static BklOrbiter *orbiter = NULL;
static gboolean running_investigator = FALSE;
static gboolean allow_start = FALSE;

/* Opts */
static gboolean replace_instance = FALSE;
static GOptionEntry bkl_entries[] = {
    { "replace", 0, 0, G_OPTION_ARG_NONE, &replace_instance,
      "Replace any currently running orbiter process", NULL },
    { NULL }
};

static void
kill_investigator (void)
{
    if (running_investigator && orbiter->investigator_pid) {
        if (kill (orbiter->investigator_pid, SIGKILL) == -1) {
            g_warning ("Error killing the investigator (%d): %s (%d)",
                       orbiter->investigator_pid, strerror (errno), errno);
            orbiter->investigator_pid = 0;
            running_investigator = FALSE;
        }
    }
}

static gboolean
processing_expired (gpointer userdata)
{
    g_warning ("Investigator has not processed anything for 10 seconds...killing");
    kill_investigator ();

    return FALSE;
}

static void
investigator_processing (DBusGProxy *proxy,
                         const char *source_name,
                         const char *current_uri,
                         gpointer    userdata)
{
    if (orbiter->investigator_processing_id > 0) {
        g_source_remove (orbiter->investigator_processing_id);
    }

    /* We take this out of the hashtable so that if it crashes/hangs
       the investigator, it won't be looked at again when we restart

       FIXME: Should we allow a uri multiple tries before failing it? */
    g_hash_table_remove (orbiter->requested_uris, current_uri);

    orbiter->investigator_processing_id = g_timeout_add_seconds
        (PROCESSING_ALLOWED_TIMEOUT, processing_expired, NULL);
}

static void G_GNUC_UNUSED
dump_metadata (gpointer key,
               gpointer value,
               gpointer userdata)
{
    g_print ("%s: %s\n", (char *) key, (char *) value);
}

#define STRING_SEPARATOR "\x01"
static GPtrArray *
string_to_string_array (const char *str)
{
    GPtrArray *array;
    char **vector;
    int i;

    if (str == NULL || *str == '\0') {
        return NULL;
    }

    vector = g_strsplit (str, STRING_SEPARATOR, 0);
    if (vector == NULL) {
        return NULL;
    }

    array = g_ptr_array_new ();
    for (i = 0; vector[i]; i++) {
        g_ptr_array_add (array, vector[i]);
    }

    g_free (vector);
    return array;
}

static BklItem *
make_audio_item (const char *uri,
                 GHashTable *metadata)
{
    BklItemAudio *item;
    const char *data;
    int number;

    item = bkl_item_audio_new ();

    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_TITLE))) {
        bkl_item_audio_set_title (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_ALBUM))) {
        bkl_item_audio_set_album (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_ARTIST))) {
        GPtrArray *array = string_to_string_array (data);
        bkl_item_audio_set_artists (item, array);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_GENRE))) {
        bkl_item_audio_set_genre (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_COMMENT))) {
        bkl_item_audio_set_comment (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_YEAR))) {
        bkl_item_audio_set_year (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_DURATION))) {
        number = atoi (data);
        bkl_item_audio_set_duration (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_TRACK))) {
        number = atoi (data);
        bkl_item_audio_set_track (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_TOTAL_TRACKS))) {
        number = atoi (data);
        bkl_item_audio_set_maxtracks (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_DISC))) {
        number = atoi (data);
        bkl_item_audio_set_disc (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_TOTAL_DISCS))) {
        number = atoi (data);
        bkl_item_audio_set_maxdiscs (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_PERFORMERS))) {
        GPtrArray *array = string_to_string_array (data);
        bkl_item_audio_set_performers (item, array);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_COPYRIGHT))) {
        bkl_item_audio_set_copyright (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_LICENSE))) {
        bkl_item_audio_set_license (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_ORGANISATION))) {
        bkl_item_audio_set_organisation (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_CONTACT))) {
        bkl_item_audio_set_contact (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_ISRC))) {
        bkl_item_audio_set_isrc (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_COMPOSER))) {
        bkl_item_audio_set_composer (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_CONDUCTOR))) {
        bkl_item_audio_set_conductor (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_AUDIO_ARTIST_IMAGE))) {
        bkl_item_audio_set_artist_image (item, data);
    }
    return (BklItem *) item;
}

static BklItem *
make_image_item (const char *uri,
                 GHashTable *metadata)
{
    BklItemImage *item;
    const char *data;
    int number;

    item = bkl_item_image_new ();

    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_TITLE))) {
        bkl_item_image_set_title (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_MODEL))) {
        bkl_item_image_set_camera_model (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_MAKE))) {
        bkl_item_image_set_camera_make (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_DATE_TIME))) {
        bkl_item_image_set_time (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_DATE_TIME_ORIGINAL))) {
        bkl_item_image_set_time_original (item, data);
    }
    if( (data = g_hash_table_lookup (metadata, METADATA_IMAGE_DATE_TIME_DIGITIZED))) {
        bkl_item_image_set_time_digitized (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_ORIENTATION))) {
        bkl_item_image_set_orientation (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_EXPOSURE_TIME))) {
        bkl_item_image_set_exposure (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_APERTURE_VALUE))) {
        bkl_item_image_set_aperture (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_METERING_MODE))) {
        bkl_item_image_set_metering_mode (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_FLASH))) {
        bkl_item_image_set_flash (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_FOCAL_LENGTH))) {
        bkl_item_image_set_focal_length (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_SHUTTER_SPEED))) {
        bkl_item_image_set_shutter_speed (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_ISO))) {
        bkl_item_image_set_iso (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_SOFTWARE))) {
        bkl_item_image_set_software (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_COMMENT))) {
        bkl_item_image_set_comment (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_USER_COMMENT))) {
        bkl_item_image_set_user_comment (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_AUTHOR))) {
        bkl_item_image_set_author (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_COPYRIGHT))) {
        bkl_item_image_set_copyright (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_KEYWORDS))) {
        bkl_item_image_set_keywords (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_SUBJECT))) {
        bkl_item_image_set_subject (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_WIDTH))) {
        number = atoi (data);
        bkl_item_image_set_width (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_IMAGE_HEIGHT))) {
        number = atoi (data);
        bkl_item_image_set_height (item, number);
    }

    return (BklItem *) item;
}

static BklItem *
make_video_item (const char *uri,
                 GHashTable *metadata)
{
    BklItemVideo *item;
    const char *data;
    int number;

    item = bkl_item_video_new ();

    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_TITLE))) {
        bkl_item_video_set_title (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_DIRECTOR))) {
        bkl_item_video_set_director (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_SERIES_NAME))) {
        bkl_item_video_set_series_name (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_SEASON))) {
        number = atoi (data);
        bkl_item_video_set_season (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_EPISODE))) {
        number = atoi (data);
        bkl_item_video_set_episode (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_YEAR))) {
        number = atoi (data);
        bkl_item_video_set_year (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_WIDTH))) {
        number = atoi (data);
        bkl_item_video_set_width (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_HEIGHT))) {
        number = atoi (data);
        bkl_item_video_set_height (item, number);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_SERIES_IMAGE))) {
        bkl_item_video_set_series_image (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_VIDEO_DISC))) {
        number = atoi (data);
        bkl_item_video_set_disc (item, number);
    }
    return (BklItem *) item;
}

BklItem *
make_broken_item (const char *uri,
                  GHashTable *metadata)
{
    BklItemBroken *item;
    const char *data;

    item = bkl_item_broken_new ();

    if ((data = g_hash_table_lookup (metadata, METADATA_BROKEN_REASON))) {
        bkl_item_broken_set_reason (item, data);
    }

    return (BklItem *) item;
}

static void
add_file_metadata (BklItem    *item,
                   GHashTable *metadata)
{
    const char *data;

    if ((data = g_hash_table_lookup (metadata, METADATA_FILE_URI))) {
        bkl_item_set_uri (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_FILE_MIMETYPE))) {
        bkl_item_set_mimetype (item, data);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_FILE_MTIME))) {
        glong mtime = strtol (data, NULL, 10);
        bkl_item_set_modification_time (item, mtime);
    }
    if ((data = g_hash_table_lookup (metadata, METADATA_FILE_SIZE))) {
        goffset size = strtoull (data, NULL, 10);
        bkl_item_set_size (item, size);
    }
}

static void
add_extended_metadata (BklItem    *item,
                       GHashTable *metadata)
{
    const char *data;

    if ((data = g_hash_table_lookup (metadata, METADATA_EXTENDED_THUMBNAIL))) {
        bkl_item_extended_set_thumbnail ((BklItemExtended *) item, data);
    }

    if ((data = g_hash_table_lookup (metadata, METADATA_EXTENDED_DATE_INDEXED))) {
        glong indexed = strtol (data, NULL, 10);
        bkl_item_extended_set_date_indexed ((BklItemExtended *) item, indexed);
    }
}

static void
investigator_completed (DBusGProxy *proxy,
                        const char *source_name,
                        const char *uri,
                        GHashTable *metadata,
                        gpointer    userdata)
{
    BklSource *source;
    BklItem *item = NULL;
    const char *type;
    GError *error = NULL;

    /* Turn off any processing request */
    if (orbiter->investigator_processing_id > 0) {
        g_source_remove (orbiter->investigator_processing_id);
        orbiter->investigator_processing_id = 0;
    }

    if (g_hash_table_size (metadata) == 0) {
        /* This wasn't a filetype we really cared about */
        return;
    }

    type = g_hash_table_lookup (metadata, METADATA_TYPE);
    if (type == NULL) {
        return;
    }

    if (g_str_equal (type, AUDIO_TYPE)) {
        item = make_audio_item (uri, metadata);
    } else if (g_str_equal (type, IMAGE_TYPE)) {
        item = make_image_item (uri, metadata);
    } else if (g_str_equal (type, VIDEO_TYPE)) {
        item = make_video_item (uri, metadata);
    } else if (g_str_equal (type, BROKEN_TYPE)) {
        item = make_broken_item (uri, metadata);
    } else if (g_str_equal (type, PLAYLIST_TYPE)) {
        return;
    } else {
        g_warning ("Unknown type: %s", type);
        return;
    }

    add_file_metadata (item, metadata);
    add_extended_metadata (item, metadata);
    /* g_hash_table_foreach (metadata, dump_metadata, NULL); */

    source = g_hash_table_lookup (orbiter->sources, source_name);
    if (source == NULL) {
        g_warning ("Error getting source: %s", source_name);
        return;
    }

    bkl_source_add_item (source, uri, item, &error);
    g_object_unref (item);
}

static gboolean
ensure_investigator (void)
{
    if (!running_investigator) {
        GError *error = NULL;
        guint start_ret;

        if (!org_freedesktop_DBus_start_service_by_name (orbiter->proxy,
                                                         INVESTIGATOR_NAME, 0,
                                                         &start_ret, &error)) {
            g_warning ("Investigator has failed to start: %s",
                       error->message);
            g_error_free (error);

            running_investigator = FALSE;
            return FALSE;
        }

        running_investigator = TRUE;

        orbiter->investigator = dbus_g_proxy_new_for_name (orbiter->connection,
                                                           INVESTIGATOR_NAME,
                                                           INVESTIGATOR_PATH,
                                                           INVESTIGATOR_IFACE);
        dbus_g_proxy_add_signal (orbiter->investigator, "Processing",
                                 G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);
        dbus_g_proxy_connect_signal (orbiter->investigator, "Processing",
                                     G_CALLBACK (investigator_processing),
                                     NULL, NULL);
        dbus_g_proxy_add_signal (orbiter->investigator, "UriCompleted",
                                 G_TYPE_STRING, G_TYPE_STRING,
                                 DBUS_TYPE_G_STRING_STRING_HASHTABLE,
                                 G_TYPE_INVALID);
        dbus_g_proxy_connect_signal (orbiter->investigator, "UriCompleted",
                                     G_CALLBACK (investigator_completed),
                                     NULL, NULL);

        if (!org_moblin_Bickley_Investigator_get_process_id
            (orbiter->investigator, &orbiter->investigator_pid, &error)) {
            g_warning ("Error getting investigator PID: %s", error->message);
            g_error_free (error);
            orbiter->investigator_pid = 0;
            return FALSE;
        }
    }

    return TRUE;
}

/* Sort pending uris according to source name */
static int
sort_pending_uris (gconstpointer a,
                   gconstpointer b)
{
    struct _PendingUri *pa, *pb;

    pa = (struct _PendingUri *) a;
    pb = (struct _PendingUri *) b;

    return strcmp (pa->source->name, pb->source->name);
}

static void
name_owner_changed (DBusGProxy *proxy,
                    const char *name,
                    const char *old_owner,
                    const char *new_owner,
                    gpointer    userdata)
{
    int size;

    if (strcmp (name, INVESTIGATOR_NAME) != 0) {
        /* Only need to listen for investigator changing */
        return;
    }

    if (new_owner && *new_owner != 0) {
        /* Investigator has just been owned so everything is good */
        return;
    }

    /* New owner is empty, so the investigator has left the bus
       for some reason. Check to see if it needs restarted */
    if (orbiter->investigator_processing_id) {
        g_source_remove (orbiter->investigator_processing_id);
        orbiter->investigator_processing_id = 0;
    }

    orbiter->investigator_pid = 0;
    g_object_unref (orbiter->investigator);
    orbiter->investigator = NULL;
    running_investigator = FALSE;

    size = g_hash_table_size (orbiter->requested_uris);
    if (size > 0) {
        GList *pending, *p;
        GPtrArray *fa = NULL;
        BklSource *current_source = NULL;
        GError *error = NULL;

        /* We were still waiting to hear back from the investigator about
           some uris, so it has not finished its work. Need to remove the
           last uri it was trying and restart it */
        ensure_investigator ();

        pending = g_hash_table_get_values (orbiter->requested_uris);
        pending = g_list_sort (pending, sort_pending_uris);

        /* Go through the list of pending uris, and collect them altogether
           according to source */
        for (p = pending; p; p = p->next) {
            struct _PendingUri *pu = (struct _PendingUri *) p->data;
            if (G_UNLIKELY (current_source == NULL)) {
                current_source = pu->source;
                fa = g_ptr_array_new ();
            }

            g_ptr_array_add (fa, pu->uri);

            if (fa->len >= 50 ||
                p->next == NULL ||
                current_source != ((struct _PendingUri *)p->next->data)->source) {
                g_ptr_array_add (fa, NULL);

                if (!org_moblin_Bickley_Investigator_investigate_uris
                    (orbiter->investigator, current_source->name,
                     (const char **) fa->pdata, fa->len, &error)) {
                    g_warning ("(%s): Error telling investigator about %d uris: %s",
                               current_source->name, fa->len,
                               error->message);
                    g_error_free (error);
                    error = NULL;
                }

                /* Don't need to free the uris in @fa as they should be
                   freed when removed from the requested_uri hashtable */
                g_ptr_array_free (fa, TRUE);
                current_source = NULL;
            }
        }
    }
}

static void
investigate_uris_reply (DBusGProxy *proxy,
                        GError     *error,
                        gpointer    data)
{
    if (error != NULL) {
        g_warning ("Error telling Investigator about uris: %s", error->message);
    }
}

/* Returns the number of files that are actually being investigated */
guint
bkl_orbiter_investigate_files (BklSource *source,
                               GPtrArray *files)
{
    GPtrArray *fa;
    int i;
    guint count;

    fa = g_ptr_array_new ();
    i = 0;
    while (i < files->len) {
        struct _PendingUri *pu;
        char *uri = files->pdata[i];
        gpointer value, key;

        /* If we're still waiting for the uri to be indexed, ignore it */
        if (g_hash_table_lookup_extended (orbiter->requested_uris, uri,
                                          &key, &value) == TRUE) {
            g_free (uri);
            i++;

            continue;
        }

        g_ptr_array_add (fa, uri);

        pu = g_slice_new (struct _PendingUri);
        pu->uri = uri;
        pu->source = source;

        g_hash_table_insert (orbiter->requested_uris, uri, pu);
        i++;
    }

    if (fa->len == 0) {
        g_ptr_array_free (fa, TRUE);
        return 0;
    }

    if (running_investigator == FALSE) {
        if (ensure_investigator () == FALSE) {
            for (i = 0; i < fa->len; i++) {
                g_free (fa->pdata[i]);
            }
            g_ptr_array_free (fa, TRUE);

            return 0;
        }
    }

    g_ptr_array_add (fa, NULL);

    org_moblin_Bickley_Investigator_investigate_uris_async
        (orbiter->investigator, source->name, (const char **) fa->pdata,
         fa->len, investigate_uris_reply, orbiter);

    count = fa->len - 1;
    /* Don't need to free the uris in @fa as they should be freed when
       removed from the requested_uri hashtable */
    g_ptr_array_free (fa, TRUE);

    return count;
}

static gboolean
dbus_init (void)
{
    guint32 request_name_ret;
    GError *error = NULL;

    /* Register the weird marshaller needed */
    dbus_g_object_register_marshaller (bkl_marshal_VOID__STRING_STRING,
                                       G_TYPE_NONE, G_TYPE_STRING,
                                       G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_object_register_marshaller (bkl_marshal_VOID__STRING_STRING_BOXED,
                                       G_TYPE_NONE, G_TYPE_STRING,
                                       G_TYPE_STRING, G_TYPE_BOXED,
                                       G_TYPE_INVALID);

    orbiter->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (orbiter->connection == NULL) {
        g_warning ("Error getting bus: %s", error->message);
        g_error_free (error);

        return FALSE;
    }

    orbiter->proxy = dbus_g_proxy_new_for_name (orbiter->connection,
                                                DBUS_SERVICE_DBUS,
                                                DBUS_PATH_DBUS,
                                                DBUS_INTERFACE_DBUS);
    if (!org_freedesktop_DBus_request_name (orbiter->proxy, ORBITER_NAME,
                                            0, &request_name_ret, &error)) {
        g_warning ("Error registering DBus: %s", error->message);
        g_error_free (error);

        return FALSE;
    }

    if (request_name_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
        DBusGProxy *other;

        /* If --replace wasn't passed on command line, then just accept
           that there is another orbiter running */
        if (replace_instance == FALSE) {
            return FALSE;
        }

        other = dbus_g_proxy_new_for_name (orbiter->connection,
                                           ORBITER_NAME,
                                           ORBITER_PATH,
                                           ORBITER_IFACE);
        if (!org_moblin_Bickley_Orbiter_replace (other, &error)) {
            g_warning ("Error shutting down Orbiter: %s", error->message);
            g_error_free (error);
            g_object_unref (other);

            return FALSE;
        }

        g_object_unref (other);

        /* The other Orbiter should be shut down now,
           so try to claim the name again */
        if (!org_freedesktop_DBus_request_name (orbiter->proxy, ORBITER_NAME,
                                                0, &request_name_ret, &error)) {
            g_warning ("Error registering DBus for second time: %s",
                       error->message);
            g_error_free (error);

            return FALSE;
        }

        /* Somehow, someone else claimed it before us, just give up. */
        if (request_name_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
            return FALSE;
        }
    }

    dbus_g_proxy_add_signal (orbiter->proxy, "NameOwnerChanged",
                             G_TYPE_STRING, G_TYPE_STRING,
                             G_TYPE_STRING, G_TYPE_INVALID);
    dbus_g_proxy_connect_signal (orbiter->proxy, "NameOwnerChanged",
                                 G_CALLBACK (name_owner_changed), NULL, NULL);

    orbiter->dbus = bkl_orbiter_dbus_new ();
    dbus_g_connection_register_g_object (orbiter->connection, ORBITER_PATH,
                                         (GObject *) orbiter->dbus);

    orbiter->bkl_dbus = bkl_dbus_new ();
    dbus_g_connection_register_g_object (orbiter->connection, BKL_DBUS_PATH,
                                         (GObject *) orbiter->bkl_dbus);

    orbiter->manager = g_object_new (BKL_TYPE_SOURCE_MANAGER, NULL);
    dbus_g_connection_register_g_object (orbiter->connection,
                                         BKL_SOURCE_MANAGER_PATH,
                                         (GObject *) orbiter->manager);
    return TRUE;
}

static void
get_more_work (gpointer key,
               gpointer value,
               gpointer userdata)
{
    BklSource *source = (BklSource *) value;
    gboolean *more_work = userdata;

    if (source->more_work) {
        *more_work |= bkl_source_do_work (source);
    }
}

static gboolean
worker (gpointer userdata)
{
    gboolean more_work = FALSE;

    g_hash_table_foreach (orbiter->sources, get_more_work, &more_work);

    if (more_work == FALSE) {
        /* Turn off the orbitor */
        orbiter->worker_id = 0;
        g_print ("Turning off worker\n");
        return FALSE;
    } else {
        /* Sleep for a bit before processing queue again */
        /* g_usleep (G_USEC_PER_SEC * 0.03); */

        return TRUE;
    }
}

void
bkl_orbiter_start_worker (void)
{
    if (orbiter->worker_id > 0) {
        return;
    }

    if (allow_start == TRUE) {
        orbiter->worker_id = g_idle_add_full (G_PRIORITY_LOW, worker,
                                              NULL, NULL);
    }
}

static void
mount_added (GVolumeMonitor *volume_monitor,
             GMount         *mount,
             gpointer        userdata)
{
    BklSource *source;
    GFile *root;
    const char *filename;
    char *path;

    root = g_mount_get_root (mount);
    source = bkl_source_removable_new (mount);
    if (source == NULL) {
        g_object_unref (root);
        return;
    }

    g_hash_table_insert (orbiter->sources, g_strdup (source->name), source);

    path = g_file_get_path (root);
    g_hash_table_insert (orbiter->removable_by_path, path, source);

    g_object_unref (root);

    g_print ("(%s) Created new mount\n", source->name);
    filename = kozo_db_get_filename (source->db->db);
    bkl_source_manager_add_source (orbiter->manager, source);

    /* bkl_orbiter_dbus_index_in_progress (orbiter->dbus, source->name); */
    bkl_orbiter_start_worker ();
}

static gboolean
remove_source_uris (gpointer key,
                    gpointer value,
                    gpointer data)
{
    BklSource *source = (BklSource *) data;
    struct _PendingUri *pu = (struct _PendingUri *) value;

    if (pu->source == source) {
        return TRUE;
    }

    return FALSE;
}

/* mount_removed is on both mount-pre-unmount and mount-removed
   so we can clean up correctly when a mount is just pulled out */
static void
mount_removed (GVolumeMonitor *volume_monitor,
               GMount         *mount,
               gpointer        userdata)
{
    BklSource *source;
    const char *filename;
    GError *error = NULL;
    GFile *root;
    char *path;

    root = g_mount_get_root (mount);
    path = g_file_get_path (root);

    source = g_hash_table_lookup (orbiter->removable_by_path, path);
    g_object_unref (root);

    if (source == NULL) {
        g_free (path);
        return;
    }

    g_hash_table_remove (orbiter->removable_by_path, path);
    g_free (path);
    g_hash_table_remove (orbiter->sources, source->name);

    filename = kozo_db_get_filename (source->db->db);

    if (orbiter->investigator) {
        if (!org_moblin_Bickley_Investigator_cancel_source
            (orbiter->investigator, source->name, &error)) {
            g_warning ("Error cancelling %s: %s", source->name, error->message);
            g_error_free (error);
        }
    }

    /* Remove the pending uris for this source */
    g_hash_table_foreach_remove (orbiter->requested_uris,
                                 (GHRFunc) remove_source_uris, source);

    bkl_source_manager_remove_source (orbiter->manager, source);

    g_object_unref (source);
}

static void
init_volume_monitor (void)
{
    GList *mounts, *m;

    orbiter->volume_monitor = g_volume_monitor_get ();
    g_signal_connect (orbiter->volume_monitor, "mount-added",
                      G_CALLBACK (mount_added), NULL);
    g_signal_connect (orbiter->volume_monitor, "mount-pre-unmount",
                      G_CALLBACK (mount_removed), NULL);
    g_signal_connect (orbiter->volume_monitor, "mount-removed",
                      G_CALLBACK (mount_removed), NULL);

    mounts = g_volume_monitor_get_mounts (orbiter->volume_monitor);
    for (m = mounts; m; m = m->next) {
        GMount *mount = m->data;

        mount_added (orbiter->volume_monitor, mount, NULL);
        g_object_unref (mount);
    }
    g_list_free (mounts);
}

static void
upnp_device_available (GUPnPControlPoint *cp,
                       GUPnPDeviceProxy  *proxy,
                       gpointer           userdata)
{
    GUPnPDeviceInfo *info = (GUPnPDeviceInfo *) proxy;
    BklSource *source;
    const char *udn, *filename;

    source = bkl_source_upnp_new (proxy);
    if (source == NULL) {
        char *name;

        name = gupnp_device_info_get_friendly_name (info);
        g_warning ("Failed to create a source for UPnP device %s (%s)",
                   name, gupnp_device_info_get_udn (info));

        g_free (name);
        return;
    }

    udn = gupnp_device_info_get_udn (info);
    g_hash_table_insert (orbiter->sources, g_strdup (udn), source);

    filename = kozo_db_get_filename (source->db->db);
    bkl_source_manager_add_source (orbiter->manager, source);

    /* bkl_orbiter_dbus_index_in_progress (orbiter->dbus, source->name); */
    bkl_orbiter_start_worker ();
}

static void
upnp_device_unavailable (GUPnPControlPoint *cp,
                         GUPnPDeviceProxy  *proxy,
                         gpointer           userdata)
{
    GUPnPDeviceInfo *info = (GUPnPDeviceInfo *) proxy;
    BklSource *source;
    const char *udn, *filename;

    udn = gupnp_device_info_get_udn (info);
    source = g_hash_table_lookup (orbiter->sources, udn);

    if (source == NULL) {
        return;
    }

    g_hash_table_remove (orbiter->sources, udn);

    filename = kozo_db_get_filename (source->db->db);
    bkl_source_manager_remove_source (orbiter->manager, source);

    g_object_unref (source);
}

static void
init_upnp_monitor (void)
{
    orbiter->upnp_monitor = gupnp_context_new (NULL, NULL, 0, NULL);
    orbiter->upnp_cp = gupnp_control_point_new
        (orbiter->upnp_monitor, "urn:schemas-upnp-org:device:MediaServer:1");
    g_signal_connect (orbiter->upnp_cp, "device-proxy-available",
                      G_CALLBACK (upnp_device_available), NULL);
    g_signal_connect (orbiter->upnp_cp, "device-proxy-unavailable",
                      G_CALLBACK (upnp_device_unavailable), NULL);

    /* Start UPnP searching */
    gssdp_resource_browser_set_active
        (GSSDP_RESOURCE_BROWSER (orbiter->upnp_cp), TRUE);
}

static gboolean
initial_wakeup_call (gpointer userdata)
{
    /* Now we've woken up we can allow the worker to do something */
    allow_start = TRUE;

    /* This will run through the sources to see if there is work to do
       and if not, it will turn itself off */
    bkl_orbiter_start_worker ();

    return FALSE;
}

static void
free_pending_uri (gpointer data)
{
    struct _PendingUri *pu = (struct _PendingUri *) data;

    g_free (pu->uri);
    g_slice_free (struct _PendingUri, pu);
}

int
main (int    argc,
      char **argv)
{
    GOptionContext *context;
    GError *error = NULL;

    g_thread_init (NULL);
    bkl_init ();

    context = g_option_context_new ("- Bickley Orbiter");
    g_option_context_add_main_entries (context, bkl_entries, "");
    g_option_context_set_ignore_unknown_options (context, TRUE);

    g_option_context_parse (context, &argc, &argv, &error);
    if (error != NULL) {
        g_warning ("Error parsing options: %s", error->message);
        g_error_free (error);
        error = NULL;
    }

    g_option_context_free (context);

    orbiter = g_new0 (BklOrbiter, 1);
    if (dbus_init () == FALSE) {
        /* Orbiter is already running, so just quit */
        return 0;
    }

    /* The key is not freed by this because it is freed by free_pending_uri */
    orbiter->requested_uris = g_hash_table_new_full (g_str_hash, g_str_equal,
                                                     NULL, free_pending_uri);
    orbiter->sources = g_hash_table_new_full (g_str_hash, g_str_equal,
                                              g_free, NULL);
    orbiter->removable_by_path = g_hash_table_new_full (g_str_hash, g_str_equal,
                                                        g_free, NULL);

    orbiter->local_source = bkl_source_gconf_new ("local-media");
    if (orbiter->local_source) {
        g_hash_table_insert (orbiter->sources, orbiter->local_source->name,
                             orbiter->local_source);

        /* The local source isn't under the manager */
        dbus_g_connection_register_g_object (orbiter->connection,
                                             BKL_LOCAL_SOURCE_PATH,
                                             G_OBJECT (orbiter->local_source));
    }

    init_volume_monitor ();

    if (g_getenv ("BKL_ORBITER_NO_UPNP") == NULL) {
        init_upnp_monitor ();
    }

    /* ... 'til Brooklyn! */
    if (g_getenv ("BKL_ORBITER_NO_SLEEP")) {
        initial_wakeup_call (NULL);
    } else {
        g_timeout_add_seconds (INITIAL_SLEEP_TIMEOUT,
                               initial_wakeup_call, NULL);
    }

    orbiter->mainloop = g_main_loop_new (NULL, TRUE);
    g_main_loop_run (orbiter->mainloop);

    kozo_shutdown ();

    return 0;
}

GList *
bkl_orbiter_get_sources (void)
{
    GList *sources;

    sources = g_hash_table_get_values (orbiter->sources);
    return sources;
}

void
bkl_orbiter_shutdown (void)
{
    g_main_loop_quit (orbiter->mainloop);
}
