<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    holiday-defs.php                                        */
/* Author:      Paul Waite                                              */
/* Description: Utility datetime functions to handle public holidays.   */
/*                                                                      */
/* ******************************************************************** */
/** @package datetime */

include_once("datetime-defs.php");

// ----------------------------------------------------------------------
/**
 * A class to handle a list of public holidays. It allows you to add
 * holidays in various ways, and also to query whether a particular
 * date-time falls on a holiday.
 */
class holidays {
  // Array of holiday timestamps..
  var $hols = array();
  
  // Last matched holiday description..
  var $matched_holiday_desc = "";
  
  // ....................................................................
  /** Constructor. Optionally pass in an associative array with the key
   * of a unique holiday description, and value as the timestamp of the
   * holiday itself.
   * @param array $hols Associative array 'Holiday desc' => timestamp
   */
  function holidays($hols="") {
    if (is_array($hols)) {
      $this->hols = $hols;
    }
  } // holidays

  // ....................................................................
  /** Add a holiday timestamp to the array. This is the internal method
   * for adding raw holiday timestamps.
   * @param string $holdesc Unique descriptive identifier for the holiday
   * @param string $holts The holiday timestamp to add
   */
  function add_holiday_ts($holdesc, $holts) {
    $this->hols[$holdesc] = $holts;
  } // add_holiday_ts
  
  // ....................................................................
  /** Add a holiday date as a date-string in the usual kinds of form
   * such as '25-12', or '25 December 2007' or '25/12/2007' etc. This
   * is a friendly method for adding general holiday dates.
   * @param string $holdesc Unique descriptive identifier for the holiday
   * @param string $holdate The holiday date as a string eg '25/12/2006'
   */
  function add_holiday($holdesc, $holdate) {
    $holts = strtotime($holdate);
    if ($holts != -1) {
      $this->add_holiday_ts($holdesc, $holts);
    }
  } // add_holiday
  
  // ....................................................................
  /** Add a holiday which is calculated as being the Nth occurence
   * of a given dayname, in a given month. This method is a common one
   * for designating National holidays, eg. 'Queen's Birthday' as
   * being the first Monday in June.
   * @param string $holdesc Unique descriptive identifier for the holiday
   * @param string $monthname Name of month eg. 'Dec' or 'December'
   * @param string $dayabbrev Dayname abbreviation eg. 'Mon', 'Tue'..
   * @param integer $n Occurence in month eg. 1, 2, 3..
   * @param integer $year Optional year (defaults to current year)
   */
  function add_holiday_relative($holdesc, $monthname, $dayabbrev, $n, $year="") {
    $holts = get_nth_day_of_month($monthname, $dayabbrev, $n, $year);
    if ($holts != -1) { 
      $this->add_holiday($holdesc, $holts);
    }
  } // add_holiday_relative
  
  // ....................................................................
  /**
  * Check if the given timestamp is the date of a public holiday. We
  * return true if it is, else false. Note: if the timestamp falls on
  * a holiday, then the holiday description is left in the class var
  * 'matched_holiday_desc' on exit. If the passed-in timestamp is not
  * given, then the method assume 'now' as the timestamp.
  * @param integer $timestamp Unix timestamp of the date to check.
  * @return boolean True if the timestamp is on a public holiday
  */
  function is_holiday($timestamp="") {
    if ($timestamp == "") {
      $timestamp = time();
    }
    $ishol = false;
    $this->matched_holiday_desc = "";
    foreach ($this->hols as $holdesc => $hol) {
      if (strstr($hol, "-")) {
        $hol_ts = strtotime($hol);
      }
      else {
        $bits = explode(" ", $hol);
        $hol_ts = get_nth_day_of_month($bits[2], $bits[0], $bits[1]);
      }
      if (is_same_day($timestamp, $hol_ts)) {
        $ishol = true;
        $this->matched_holiday_desc = $holdesc;
        break;
      }
    } // foreach
  
    return $ishol;
  } // is_holiday

} // class holidays

// ----------------------------------------------------------------------
/**
 * Return true if the two timestamps are on the same day of the calendar.
 * @return boolean True if timestamps lie on the very same day.
 */
function is_same_day($ts1, $ts2) {
  $dt1 = getdate($ts1);
  $dt2 = getdate($ts2);
  return ($dt1["mday"] == $dt2["mday"]
       && $dt1["mon"]  == $dt2["mon"]
       && $dt1["year"] == $dt2["year"]
       );
} // same_day

// ----------------------------------------------------------------------
/**
* Return the timestamp of the date for the Nth dayname in the given
* month. Returns -1 if error encountered.
* @param string $monthname Month the day should be in
* @param string $dayabbrev The 3-char abbrev: Eg. "Mon", "Tue", ..
* @param integer $n If you want the 4th day, then this would be 4
* @param integer $year Optional year, otherwise current year
* @return integer Timestamp of date, or -1 if error.
*/
function get_nth_day_of_month($monthname, $dayabbrev, $n, $year="") {
  global $daywk;

  if ($year == "") {
    $now = getdate();
    $year = $now["year"];
  }
 
  // Get day index, or default to 1 (Monday)..
  if (isset($daywk[$dayabbrev])) {
    $dayno = $daywk[$dayabbrev];
  }
  else {
    $dayno = 1;
  }
  
  // Get daynum (0=Sun, 1=Mon..)
  $start_month_ts = strtotime("1st $monthname $year");
  $start_month_dayno = strftime("%w", $start_month_ts); 
  $inc = ($dayno - $start_month_dayno + 7) % 7;
  
  $n -= 1;
  $required_date_ts = strtotime("1st $monthname $year + $inc days + $n weeks");
  return $required_date_ts;
} // get_nth_day_of_month

// ----------------------------------------------------------------------
/** Get the days of Easter for the given year (defaults to current year)
 * as an associative array of values, with the following arrangement:
 *   returned_array['Good Friday'] = timestamp of Good Friday 
 *   returned_array['Easter Saturday'] = timestamp of Easter Saturday 
 *   returned_array['Easter Sunday'] = timestamp of Easter Sunday 
 *   returned_array['Easter Monday'] = timestamp of Easter Monday 
 */
function get_easter_holidays($year="") {
  global $monthnames;
  if ($year == "") {
    $year = yearNow();
  }
  $c = (int) ($year / 100);
  $n = $year - 19 * ((int) ($year / 19));
  $k = (int) (($c - 17) / 25);
  $i = $c - ((int) ($c / 4)) - ((int) (($c - $k) / 3)) + (19 * $n) + 15;
  $i = $i - 30 * ((int) ($i / 30));
  $i = $i - ((int) (i / 28)) * (1 - ((int) ($i / 28)) * ((int) (29 / ($i + 1))) * ((int) ((21 - $n) / 11)) );
  $j = $year + ((int) ($year / 4)) + $i + 2 - $c + ((int) ($c / 4));
  $j = $j - 7 * ((int) ($j / 7));
  $l = $i - $j;
  $mth = 3 + ((int) (($l + 40) / 44));
  $monthname = $monthnames[$mth];
  $day = $l + 28 - 31 * ((int) ($mth / 4));
  $easter_sunday_ts = strtotime("$day $monthname $year");

  $easter = array();
  $easter["Good Friday"]     = $easter_sunday_ts - (2 * 86400);
  $easter["Easter Saturday"] = $easter_sunday_ts - (1 * 86400); 
  $easter["Easter Sunday"]   = $easter_sunday_ts;
  $easter["Easter Monday"]   = $easter_sunday_ts + (1 * 86400);
  
  // Return array of timestamps..
  return $easter;
  
} // get_easter_holidays

// ----------------------------------------------------------------------
?>